/*
 * WangXun RP1000/RP2000/FF50XX PCI Express Linux driver
 * Copyright (c) 2015 - 2025 Beijing WangXun Technology Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * The full GNU General Public License is included in this distribution in
 * the file called "COPYING".
 *
 * based on txgbe_ethtool.c, Copyright(c) 1999 - 2017 Intel Corporation.
 * Contact Information:
 * Linux NICS <linux.nics@intel.com>
 * e1000-devel Mailing List <e1000-devel@lists.sourceforge.net>
 * Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497
 */

/* ethtool support for txgbe */

#include <linux/types.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/netdevice.h>
#include <linux/ethtool.h>
#include <linux/vmalloc.h>
#include <linux/highmem.h>
#include <linux/firmware.h>
#ifdef SIOCETHTOOL
#include <asm/uaccess.h>

#include "txgbe.h"
#include "txgbe_hw.h"
#if defined(ETHTOOL_GMODULEINFO)||defined(HAVE_ETHTOOL_SET_PHYS_ID)
#include "txgbe_phy.h"
#include "txgbe_e56.h"
#endif
#ifdef HAVE_ETHTOOL_GET_TS_INFO
#include <linux/net_tstamp.h>
#endif

#ifndef ETH_GSTRING_LEN
#define ETH_GSTRING_LEN 32
#endif

#define TXGBE_ALL_RAR_ENTRIES 16

#ifdef ETHTOOL_OPS_COMPAT
#include "kcompat_ethtool.c"
#endif

#include "txgbe_xsk.h"

#define ETHTOOL_LINK_MODE_SPEED_MASK	0xfffe903f

#ifdef ETHTOOL_GSTATS
struct txgbe_stats {
	char stat_string[ETH_GSTRING_LEN];
	int sizeof_stat;
	int stat_offset;
};

#define TXGBE_NETDEV_STAT(_net_stat) { \
	.stat_string = #_net_stat, \
	.sizeof_stat = sizeof_field(struct net_device_stats, _net_stat), \
	.stat_offset = offsetof(struct net_device_stats, _net_stat) \
}
static const struct txgbe_stats txgbe_gstrings_net_stats[] = {
	TXGBE_NETDEV_STAT(rx_packets),
	TXGBE_NETDEV_STAT(tx_packets),
	TXGBE_NETDEV_STAT(rx_bytes),
	TXGBE_NETDEV_STAT(tx_bytes),
	TXGBE_NETDEV_STAT(rx_errors),
	TXGBE_NETDEV_STAT(tx_errors),
	TXGBE_NETDEV_STAT(rx_dropped),
	TXGBE_NETDEV_STAT(tx_dropped),
	TXGBE_NETDEV_STAT(collisions),
	TXGBE_NETDEV_STAT(rx_over_errors),
	TXGBE_NETDEV_STAT(rx_crc_errors),
	TXGBE_NETDEV_STAT(rx_frame_errors),
	TXGBE_NETDEV_STAT(rx_fifo_errors),
	TXGBE_NETDEV_STAT(rx_missed_errors),
	TXGBE_NETDEV_STAT(tx_aborted_errors),
	TXGBE_NETDEV_STAT(tx_carrier_errors),
	TXGBE_NETDEV_STAT(tx_fifo_errors),
	TXGBE_NETDEV_STAT(tx_heartbeat_errors),
};

#define TXGBE_STAT(_name, _stat) { \
	.stat_string = _name, \
	.sizeof_stat = sizeof_field(struct txgbe_adapter, _stat), \
	.stat_offset = offsetof(struct txgbe_adapter, _stat) \
}
static struct txgbe_stats txgbe_gstrings_stats[] = {
	TXGBE_STAT("rx_pkts_nic", stats.gprc),
	TXGBE_STAT("tx_pkts_nic", stats.gptc),
	TXGBE_STAT("rx_bytes_nic", stats.gorc),
	TXGBE_STAT("tx_bytes_nic", stats.gotc),
	TXGBE_STAT("lsc_int", lsc_int),
	TXGBE_STAT("tx_busy", tx_busy),
	TXGBE_STAT("non_eop_descs", non_eop_descs),
	TXGBE_STAT("rx_broadcast", stats.bprc),
	TXGBE_STAT("tx_broadcast", stats.bptc),
	TXGBE_STAT("rx_multicast", stats.mprc),
	TXGBE_STAT("tx_multicast", stats.mptc),
	TXGBE_STAT("rx_mac_good", stats.tpr),
	TXGBE_STAT("rdb_pkts", stats.rdpc),
	TXGBE_STAT("rdb_drop", stats.rddc),
	TXGBE_STAT("tdm_pkts", stats.tdmpc),
	TXGBE_STAT("tdm_drop", stats.tdmdc),
	TXGBE_STAT("tdb_pkts", stats.tdbpc),
	TXGBE_STAT("rx_parser_pkts", stats.psrpc),
	TXGBE_STAT("rx_parser_drop", stats.psrdc),
	TXGBE_STAT("lsec_untag_pkts", stats.untag),
	TXGBE_STAT("rx_no_buffer_count", stats.rnbc[0]),
	TXGBE_STAT("tx_timeout_count", tx_timeout_count),
	TXGBE_STAT("tx_restart_queue", restart_queue),
	TXGBE_STAT("rx_long_length_count", stats.roc),
	TXGBE_STAT("rx_short_length_count", stats.ruc),
	TXGBE_STAT("tx_flow_control_xon", stats.lxontxc),
	TXGBE_STAT("rx_flow_control_xon", stats.lxonrxc),
	TXGBE_STAT("tx_flow_control_xoff", stats.lxofftxc),
	TXGBE_STAT("rx_flow_control_xoff", stats.lxoffrxc),
	TXGBE_STAT("rx_csum_offload_good_count", hw_csum_rx_good),
	TXGBE_STAT("rx_csum_offload_errors", hw_csum_rx_error),
	TXGBE_STAT("alloc_rx_page_failed", alloc_rx_page_failed),
	TXGBE_STAT("alloc_rx_buff_failed", alloc_rx_buff_failed),
#ifndef TXGBE_NO_LRO
	TXGBE_STAT("lro_aggregated", lro_stats.coal),
	TXGBE_STAT("lro_flushed", lro_stats.flushed),
#endif /* TXGBE_NO_LRO */
	TXGBE_STAT("rx_no_dma_resources", hw_rx_no_dma_resources),
	TXGBE_STAT("hw_rsc_aggregated", rsc_total_count),
	TXGBE_STAT("hw_rsc_flushed", rsc_total_flush),
#ifdef HAVE_TX_MQ
	TXGBE_STAT("fdir_match", stats.fdirmatch),
	TXGBE_STAT("fdir_miss", stats.fdirmiss),
	TXGBE_STAT("fdir_overflow", fdir_overflow),
#endif /* HAVE_TX_MQ */
#if IS_ENABLED(CONFIG_FCOE)
	TXGBE_STAT("fcoe_bad_fccrc", stats.fccrc),
	TXGBE_STAT("fcoe_last_errors", stats.fclast),
	TXGBE_STAT("rx_fcoe_dropped", stats.fcoerpdc),
	TXGBE_STAT("rx_fcoe_packets", stats.fcoeprc),
	TXGBE_STAT("rx_fcoe_dwords", stats.fcoedwrc),
	TXGBE_STAT("fcoe_noddp", stats.fcoe_noddp),
	TXGBE_STAT("fcoe_noddp_ext_buff", stats.fcoe_noddp_ext_buff),
	TXGBE_STAT("tx_fcoe_packets", stats.fcoeptc),
	TXGBE_STAT("tx_fcoe_dwords", stats.fcoedwtc),
#endif /* CONFIG_FCOE */
	TXGBE_STAT("os2bmc_rx_by_bmc", stats.o2bgptc),
	TXGBE_STAT("os2bmc_tx_by_bmc", stats.b2ospc),
	TXGBE_STAT("os2bmc_tx_by_host", stats.o2bspc),
	TXGBE_STAT("os2bmc_rx_by_host", stats.b2ogprc),
#ifdef HAVE_PTP_1588_CLOCK
	TXGBE_STAT("tx_hwtstamp_timeouts", tx_hwtstamp_timeouts),
	TXGBE_STAT("rx_hwtstamp_cleared", rx_hwtstamp_cleared),
#endif /* HAVE_PTP_1588_CLOCK */
};

/* txgbe allocates num_tx_queues and num_rx_queues symmetrically so
 * we set the num_rx_queues to evaluate to num_tx_queues. This is
 * used because we do not have a good way to get the max number of
 * rx queues with CONFIG_RPS disabled.
 */
#ifdef HAVE_TX_MQ
#ifdef HAVE_NETDEV_SELECT_QUEUE
#define TXGBE_NUM_RX_QUEUES netdev->num_tx_queues
#define TXGBE_NUM_TX_QUEUES netdev->num_tx_queues
#else
#define TXGBE_NUM_RX_QUEUES adapter->indices
#define TXGBE_NUM_TX_QUEUES adapter->indices
#endif /* HAVE_NETDEV_SELECT_QUEUE */
#else /* HAVE_TX_MQ */
#define TXGBE_NUM_RX_QUEUES 1
#define TXGBE_NUM_TX_QUEUES ( \
		((struct txgbe_adapter *)netdev_priv(netdev))->num_tx_queues)
#endif /* HAVE_TX_MQ */

#define TXGBE_QUEUE_STATS_LEN ( \
		(TXGBE_NUM_TX_QUEUES + TXGBE_NUM_RX_QUEUES) * \
		(sizeof(struct txgbe_queue_stats) / sizeof(u64)))
#define TXGBE_GLOBAL_STATS_LEN  ARRAY_SIZE(txgbe_gstrings_stats)
#define TXGBE_NETDEV_STATS_LEN  ARRAY_SIZE(txgbe_gstrings_net_stats)
#define TXGBE_PB_STATS_LEN ( \
		(sizeof(((struct txgbe_adapter *)0)->stats.pxonrxc) + \
		 sizeof(((struct txgbe_adapter *)0)->stats.pxontxc) + \
		 sizeof(((struct txgbe_adapter *)0)->stats.pxoffrxc) + \
		 sizeof(((struct txgbe_adapter *)0)->stats.pxofftxc)) \
		/ sizeof(u64))
#define TXGBE_STATS_LEN (TXGBE_GLOBAL_STATS_LEN + \
			 TXGBE_NETDEV_STATS_LEN + \
			 TXGBE_PB_STATS_LEN + \
			 TXGBE_QUEUE_STATS_LEN)

#endif /* ETHTOOL_GSTATS */
#ifdef ETHTOOL_TEST
static const char txgbe_gstrings_test[][ETH_GSTRING_LEN] = {
	"Register test  (offline)", "Eeprom test    (offline)",
	"Interrupt test (offline)", "Loopback test  (offline)",
	"Link test   (on/offline)"
};
#define TXGBE_TEST_LEN  (sizeof(txgbe_gstrings_test) / ETH_GSTRING_LEN)
#endif /* ETHTOOL_TEST */


#ifdef HAVE_ETHTOOL_GET_SSET_COUNT
struct txgbe_priv_flags {
	char flag_string[ETH_GSTRING_LEN];
	u64 flag;
	bool read_only;
};

#define TXGBE_PRIV_FLAG(_name, _flag, _read_only) { \
	.flag_string = _name, \
	.flag = _flag, \
	.read_only = _read_only, \
}

static const struct txgbe_priv_flags txgbe_gstrings_priv_flags[] = {
	TXGBE_PRIV_FLAG("lldp", TXGBE_ETH_PRIV_FLAG_LLDP, 0),
#ifdef HAVE_SWIOTLB_SKIP_CPU_SYNC
	TXGBE_PRIV_FLAG("legacy-rx", TXGBE_ETH_PRIV_FLAG_LEGACY_RX, 0),
#endif
};

#define TXGBE_PRIV_FLAGS_STR_LEN ARRAY_SIZE(txgbe_gstrings_priv_flags)

#endif /* HAVE_ETHTOOL_GET_SSET_COUNT */

/* currently supported speeds for 10G */
#define ADVERTISED_MASK_10G (SUPPORTED_10000baseT_Full | \
		SUPPORTED_10000baseKX4_Full | SUPPORTED_10000baseKR_Full)

#define txgbe_isbackplane(type)  \
			((type == txgbe_media_type_backplane) ? true : false)

#ifdef ETHTOOL_GLINKSETTINGS
static int txgbe_set_advertising_1g_10gtypes(struct txgbe_hw *hw,
			struct ethtool_link_ksettings *cmd, u32 advertised_speed)
{
	switch (hw->phy.sfp_type) {
	case txgbe_sfp_type_da_cu:
	case txgbe_sfp_type_da_act_lmt_core0:
	case txgbe_sfp_type_da_act_lmt_core1:
	case txgbe_sfp_type_da_cu_core0:
	case txgbe_sfp_type_da_cu_core1:
	case txgbe_sfp_type_srlr_core0:
	case txgbe_sfp_type_srlr_core1:
		if (advertised_speed & TXGBE_LINK_SPEED_10GB_FULL) {
		ethtool_link_ksettings_add_link_mode(cmd, advertising, 
							 10000baseSR_Full);
		ethtool_link_ksettings_add_link_mode(cmd, advertising,
						     10000baseLR_Full);
		}
		if (advertised_speed & TXGBE_LINK_SPEED_1GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising, 
										 1000baseX_Full);
		}
		break;
	case txgbe_sfp_type_sr:
	case txgbe_sfp_type_25g_sr_core0:
	case txgbe_sfp_type_25g_sr_core1:
		if (advertised_speed & TXGBE_LINK_SPEED_10GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising, 
								 10000baseSR_Full);
		}
		if (advertised_speed & TXGBE_LINK_SPEED_1GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising, 
										 1000baseX_Full);
		}
		break;
	case txgbe_sfp_type_lr:
	case txgbe_sfp_type_25g_lr_core0:
	case txgbe_sfp_type_25g_lr_core1:
		if (advertised_speed & TXGBE_LINK_SPEED_10GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising, 
								 10000baseLR_Full);
		}
		if (advertised_speed & TXGBE_LINK_SPEED_1GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising, 
										 1000baseX_Full);
		}
		break;
	case txgbe_sfp_type_1g_cu_core0:
	case txgbe_sfp_type_1g_cu_core1:
		ethtool_link_ksettings_add_link_mode(cmd, advertising, 
							 1000baseT_Full);
		break;
	case txgbe_sfp_type_1g_sx_core0:
	case txgbe_sfp_type_1g_sx_core1:
	case txgbe_sfp_type_1g_lx_core0:
	case txgbe_sfp_type_1g_lx_core1:
		ethtool_link_ksettings_add_link_mode(cmd, advertising, 
							 1000baseX_Full);
		break;
	default:
		ethtool_link_ksettings_add_link_mode(cmd, advertising, 
							 10000baseT_Full);
		break;
	}

	return 0;
}
			
static int txgbe_set_supported_1g_10gtypes(struct txgbe_hw *hw,
			struct ethtool_link_ksettings *cmd)
{	
	switch (hw->phy.sfp_type) {
	case txgbe_sfp_type_da_cu:
	case txgbe_sfp_type_da_act_lmt_core0:
	case txgbe_sfp_type_da_act_lmt_core1:
	case txgbe_sfp_type_da_cu_core0:
	case txgbe_sfp_type_da_cu_core1:
	case txgbe_sfp_type_srlr_core0:
	case txgbe_sfp_type_srlr_core1:
		ethtool_link_ksettings_add_link_mode(cmd, supported, 
							 10000baseSR_Full);
		ethtool_link_ksettings_add_link_mode(cmd, supported,
						     10000baseLR_Full);
		break;
	case txgbe_sfp_type_sr:
	case txgbe_sfp_type_25g_sr_core0:
	case txgbe_sfp_type_25g_sr_core1:
		ethtool_link_ksettings_add_link_mode(cmd, supported, 
							 10000baseSR_Full);
		break;
	case txgbe_sfp_type_lr:
	case txgbe_sfp_type_25g_lr_core0:
	case txgbe_sfp_type_25g_lr_core1:
		ethtool_link_ksettings_add_link_mode(cmd, supported, 
							 10000baseLR_Full);
		break;
	case txgbe_sfp_type_1g_cu_core0:
	case txgbe_sfp_type_1g_cu_core1:
		ethtool_link_ksettings_add_link_mode(cmd, supported, 
							 1000baseT_Full);
		break;
	case txgbe_sfp_type_1g_sx_core0:
	case txgbe_sfp_type_1g_sx_core1:
	case txgbe_sfp_type_1g_lx_core0:
	case txgbe_sfp_type_1g_lx_core1:
		ethtool_link_ksettings_add_link_mode(cmd, supported, 
							 1000baseX_Full);
		break;
	default:
		ethtool_link_ksettings_add_link_mode(cmd, supported, 
							 10000baseT_Full);
		break;
	}

	return 0;
}

static int txgbe_get_link_ksettings(struct net_device *netdev,
				    struct ethtool_link_ksettings *cmd)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 supported_link;
	u32 link_speed = 0;
	bool autoneg = false;
	bool link_up;

	ethtool_link_ksettings_zero_link_mode(cmd, supported);
	ethtool_link_ksettings_zero_link_mode(cmd, advertising);

	TCALL(hw, mac.ops.get_link_capabilities, &supported_link, &autoneg);

	if((hw->subsystem_device_id & 0xF0) == TXGBE_ID_KR_KX_KX4)
		autoneg = adapter->backplane_an ? 1:0;
	else if((hw->subsystem_device_id & 0xF0) == TXGBE_ID_MAC_SGMII)
		autoneg = adapter->autoneg?1:0;

	/* set the supported link speeds */
	if (hw->phy.media_type == txgbe_media_type_copper) {
		if (supported_link & TXGBE_LINK_SPEED_10GB_FULL)
			ethtool_link_ksettings_add_link_mode(cmd, supported,
								 10000baseT_Full);
		if (supported_link & TXGBE_LINK_SPEED_1GB_FULL)
			ethtool_link_ksettings_add_link_mode(cmd, supported,
								1000baseT_Full);
		if (supported_link & TXGBE_LINK_SPEED_100_FULL)
			ethtool_link_ksettings_add_link_mode(cmd, supported,
								 100baseT_Full);

		if (supported_link & TXGBE_LINK_SPEED_10_FULL)
			ethtool_link_ksettings_add_link_mode(cmd, supported,
								 10baseT_Full);
	} else if (hw->phy.media_type == txgbe_media_type_fiber_qsfp) {
		if (supported_link & TXGBE_LINK_SPEED_40GB_FULL)
			ethtool_link_ksettings_add_link_mode(cmd, supported,
					 40000baseSR4_Full);
	} else if (hw->phy.media_type == txgbe_media_type_fiber) {
		if (supported_link & TXGBE_LINK_SPEED_25GB_FULL)
			ethtool_link_ksettings_add_link_mode(cmd, supported,
					 25000baseSR_Full);

		if ((supported_link & TXGBE_LINK_SPEED_10GB_FULL) ||
			(supported_link & TXGBE_LINK_SPEED_1GB_FULL))
			txgbe_set_supported_1g_10gtypes(hw, cmd);
		if (hw->phy.multispeed_fiber && hw->mac.type == txgbe_mac_sp)
			ethtool_link_ksettings_add_link_mode(cmd, supported,
							 1000baseX_Full);
	} else {
		switch (hw->phy.link_mode) {
		case TXGBE_PHYSICAL_LAYER_10GBASE_KX4:
			ethtool_link_ksettings_add_link_mode(cmd, supported,
						     10000baseKX4_Full);
			break;
		case TXGBE_PHYSICAL_LAYER_10GBASE_KR:
			ethtool_link_ksettings_add_link_mode(cmd, supported,
						     10000baseKR_Full);
			break;
		case TXGBE_PHYSICAL_LAYER_1000BASE_KX:
			ethtool_link_ksettings_add_link_mode(cmd, supported,
						     1000baseKX_Full);
			break;
		default:
			ethtool_link_ksettings_add_link_mode(cmd, supported,
						     10000baseKR_Full);
			ethtool_link_ksettings_add_link_mode(cmd, supported,
						     10000baseKX4_Full);
			break;
		}
	}

		/* set the advertised speeds */
	if (hw->phy.autoneg_advertised) {
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_40GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising,
						 40000baseSR4_Full);
		}
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_25GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising,
						 25000baseSR_Full);
		}
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_10GB_FULL) {
			if (hw->phy.media_type == txgbe_media_type_copper) {
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 10000baseT_Full);
			} else if (hw->phy.media_type == txgbe_media_type_fiber) {
				txgbe_set_advertising_1g_10gtypes(hw, cmd,
					         hw->phy.autoneg_advertised);
			} else {
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 10000baseKR_Full);
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 10000baseKX4_Full);
			}
		}
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_1GB_FULL) {
			if (hw->phy.media_type == txgbe_media_type_copper)
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 1000baseT_Full);
			else if (hw->phy.media_type == txgbe_media_type_fiber)
				txgbe_set_advertising_1g_10gtypes(hw, cmd,
							 hw->phy.autoneg_advertised);
			else
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 1000baseKX_Full);
		}
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_100_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 100baseT_Full);
		}
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_10_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising, 
							 10baseT_Full);
		}
	} else {
		if (supported_link & TXGBE_LINK_SPEED_40GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising,
					 40000baseSR4_Full);
		}
		if (supported_link & TXGBE_LINK_SPEED_25GB_FULL) {
			ethtool_link_ksettings_add_link_mode(cmd, advertising,
					 25000baseSR_Full);
		}
		if (supported_link & TXGBE_LINK_SPEED_10GB_FULL) {
			if (hw->phy.media_type == txgbe_media_type_copper) {
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
						 10000baseT_Full);
			} else if (hw->phy.media_type == txgbe_media_type_fiber) {
				txgbe_set_advertising_1g_10gtypes(hw, cmd,
					         TXGBE_LINK_SPEED_10GB_FULL);
			} else {
				switch (hw->phy.link_mode) {
				case TXGBE_PHYSICAL_LAYER_10GBASE_KX4:
					ethtool_link_ksettings_add_link_mode(cmd, advertising,
									10000baseKX4_Full);
					break;
				case TXGBE_PHYSICAL_LAYER_10GBASE_KR:
					ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 10000baseKR_Full);
					break;
				default:
					ethtool_link_ksettings_add_link_mode(cmd, advertising,
									10000baseKR_Full);
					ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 10000baseKX4_Full);
					break;
				}
			}
		}
		if (supported_link & TXGBE_LINK_SPEED_1GB_FULL) {
			if (hw->phy.media_type == txgbe_media_type_copper)
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
						 1000baseT_Full);
			else if (hw->phy.media_type == txgbe_media_type_fiber)
				txgbe_set_advertising_1g_10gtypes(hw, cmd,
					         TXGBE_LINK_SPEED_1GB_FULL);
			else
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
							 1000baseKX_Full);
		}
		if (supported_link & TXGBE_LINK_SPEED_100_FULL) {
			if (hw->phy.media_type == txgbe_media_type_copper)
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
						 100baseT_Full);
		}
		if (supported_link & TXGBE_LINK_SPEED_10_FULL) {
			if (hw->phy.media_type == txgbe_media_type_copper)
				ethtool_link_ksettings_add_link_mode(cmd, advertising,
						 10baseT_Full);
		}
	}

	if (autoneg) {
		ethtool_link_ksettings_add_link_mode(cmd, supported, Autoneg);
		ethtool_link_ksettings_add_link_mode(cmd, advertising, Autoneg);
		cmd->base.autoneg = AUTONEG_ENABLE;
	} else
		cmd->base.autoneg = AUTONEG_DISABLE;

	/* Determine the remaining settings based on the PHY type. */
	switch (adapter->hw.phy.type) {
	case txgbe_phy_tn:
	case txgbe_phy_aq:
	case txgbe_phy_cu_unknown:
		ethtool_link_ksettings_add_link_mode(cmd, supported, TP);
		ethtool_link_ksettings_add_link_mode(cmd, advertising, TP);
		cmd->base.port = PORT_TP;
		break;
	case txgbe_phy_qt:
		ethtool_link_ksettings_add_link_mode(cmd, supported, FIBRE);
		ethtool_link_ksettings_add_link_mode(cmd, advertising, FIBRE);
		cmd->base.port = PORT_FIBRE;
		break;
	case txgbe_phy_nl:
	case txgbe_phy_sfp_passive_tyco:
	case txgbe_phy_sfp_passive_unknown:
	case txgbe_phy_sfp_active_unknown:
	case txgbe_phy_sfp_ftl_active:
	case txgbe_phy_sfp_ftl:
	case txgbe_phy_sfp_avago:
	case txgbe_phy_sfp_intel:
	case txgbe_phy_sfp_unknown:
		switch (adapter->hw.phy.sfp_type) {
			/* SFP+ devices, further checking needed */
		case txgbe_sfp_type_da_cu:
		case txgbe_sfp_type_da_cu_core0:
		case txgbe_sfp_type_da_cu_core1:
		case txgbe_qsfp_type_40g_cu_core0:
		case txgbe_qsfp_type_40g_cu_core1:
			ethtool_link_ksettings_add_link_mode(cmd, supported, FIBRE);
			ethtool_link_ksettings_add_link_mode(cmd, advertising, FIBRE);
			cmd->base.port = PORT_DA;
			break;
		case txgbe_sfp_type_sr:
		case txgbe_sfp_type_lr:
		case txgbe_sfp_type_srlr_core0:
		case txgbe_sfp_type_srlr_core1:
		case txgbe_sfp_type_1g_sx_core0:
		case txgbe_sfp_type_1g_sx_core1:
		case txgbe_sfp_type_1g_lx_core0:
		case txgbe_sfp_type_1g_lx_core1:
		case txgbe_sfp_type_da_act_lmt_core0:
		case txgbe_sfp_type_da_act_lmt_core1:
		case txgbe_sfp_type_25g_sr_core0:
		case txgbe_sfp_type_25g_sr_core1:
		case txgbe_sfp_type_25g_lr_core0:
		case txgbe_sfp_type_25g_lr_core1:
		case txgbe_sfp_type_25g_aoc_core0:
		case txgbe_sfp_type_25g_aoc_core1:
		case txgbe_qsfp_type_40g_sr_core0:
		case txgbe_qsfp_type_40g_sr_core1:
		case txgbe_qsfp_type_40g_lr_core0:
		case txgbe_qsfp_type_40g_lr_core1:
		case txgbe_qsfp_type_40g_active_core0:
		case txgbe_qsfp_type_40g_active_core1:
			ethtool_link_ksettings_add_link_mode(cmd, supported, FIBRE);
			ethtool_link_ksettings_add_link_mode(cmd, advertising, FIBRE);
			cmd->base.port = PORT_FIBRE;
			break;
		case txgbe_sfp_type_not_present:
			ethtool_link_ksettings_add_link_mode(cmd, supported, FIBRE);
			ethtool_link_ksettings_add_link_mode(cmd, advertising, FIBRE);
			cmd->base.port = PORT_NONE;
			break;
		case txgbe_sfp_type_1g_cu_core0:
		case txgbe_sfp_type_1g_cu_core1:
			ethtool_link_ksettings_add_link_mode(cmd, supported, TP);
			ethtool_link_ksettings_add_link_mode(cmd, advertising, TP);
			cmd->base.port = PORT_TP;
			break;
		case txgbe_sfp_type_unknown:
		default:
			ethtool_link_ksettings_add_link_mode(cmd, supported, FIBRE);
			ethtool_link_ksettings_add_link_mode(cmd, advertising, FIBRE);
			cmd->base.port = PORT_OTHER;
			break;
		}
		break;
	case txgbe_phy_xaui:
		ethtool_link_ksettings_add_link_mode(cmd, supported, TP);
		ethtool_link_ksettings_add_link_mode(cmd, advertising, TP);
		cmd->base.port = PORT_TP;
		break;
	case txgbe_phy_unknown:
	case txgbe_phy_generic:
	case txgbe_phy_sfp_unsupported:
	default:
		ethtool_link_ksettings_add_link_mode(cmd, supported, FIBRE);
		ethtool_link_ksettings_add_link_mode(cmd, advertising, FIBRE);
		cmd->base.port = PORT_OTHER;
		break;
	}

	if ((hw->subsystem_device_id & 0xF0) == TXGBE_ID_XAUI &&
		(hw->subsystem_device_id & TXGBE_NCSI_MASK) == TXGBE_NCSI_SUP) {
		/* only continue if link was up previously */
		if (!netif_carrier_ok(netdev)) {
			cmd->base.speed = -1;
			cmd->base.duplex = -1;

			return 0;
		}
	}
	if (!in_interrupt()) {
		TCALL(hw, mac.ops.check_link, &link_speed, &link_up, false);
	} else {
		/*
		 * this case is a special workaround for RHEL5 bonding
		 * that calls this routine from interrupt context
		 */
		link_speed = adapter->link_speed;
		link_up = adapter->link_up;
	}

	/* Indicate pause support */
	ethtool_link_ksettings_add_link_mode(cmd, supported, Pause);

	switch (hw->fc.requested_mode) {
	case txgbe_fc_full:
		ethtool_link_ksettings_add_link_mode(cmd, advertising, Pause);
		break;
	case txgbe_fc_rx_pause:
		ethtool_link_ksettings_add_link_mode(cmd, advertising, Pause);
		ethtool_link_ksettings_add_link_mode(cmd, advertising,
						     Asym_Pause);
		break;
	case txgbe_fc_tx_pause:
		ethtool_link_ksettings_add_link_mode(cmd, advertising,
						     Asym_Pause);
		break;
	default:
		ethtool_link_ksettings_del_link_mode(cmd, advertising, Pause);
		ethtool_link_ksettings_del_link_mode(cmd, advertising,
						     Asym_Pause);
	}

	if (link_up) {
		switch (link_speed) {
		case TXGBE_LINK_SPEED_40GB_FULL:
			cmd->base.speed = SPEED_40000;
			break;
		case TXGBE_LINK_SPEED_25GB_FULL:
			cmd->base.speed = SPEED_25000;
			break;
		case TXGBE_LINK_SPEED_10GB_FULL:
			cmd->base.speed = SPEED_10000;
			break;
		case TXGBE_LINK_SPEED_1GB_FULL:
			cmd->base.speed = SPEED_1000;
			break;
		case TXGBE_LINK_SPEED_100_FULL:
			cmd->base.speed = SPEED_100;
			break;
		case TXGBE_LINK_SPEED_10_FULL:
			cmd->base.speed = SPEED_10;
			break;
		default:
			break;
			}
		cmd->base.duplex = DUPLEX_FULL;
	} else {
		cmd->base.speed = -1;
		cmd->base.duplex = -1;
	}

	if (!netif_carrier_ok(netdev)) {
		cmd->base.speed = -1;
		cmd->base.duplex = -1;
	}

#ifdef ETHTOOL_GFECPARAM
	if (hw->mac.type == txgbe_mac_aml) {
		ethtool_link_ksettings_add_link_mode(cmd, supported, FEC_NONE);
		ethtool_link_ksettings_add_link_mode(cmd, supported, FEC_RS);
		ethtool_link_ksettings_add_link_mode(cmd, supported, FEC_BASER);
		if (adapter->fec_link_mode & TXGBE_PHY_FEC_OFF)
			ethtool_link_ksettings_add_link_mode(cmd, advertising, FEC_NONE);
		if (adapter->fec_link_mode & TXGBE_PHY_FEC_RS)
			ethtool_link_ksettings_add_link_mode(cmd, advertising, FEC_RS);
		if (adapter->fec_link_mode & TXGBE_PHY_FEC_BASER)
			ethtool_link_ksettings_add_link_mode(cmd, advertising, FEC_BASER);
	}
#endif

	if(!adapter->autoneg)
		ethtool_link_ksettings_del_link_mode(cmd, advertising, Autoneg);
	else
		ethtool_link_ksettings_add_link_mode(cmd, advertising, Autoneg);
	cmd->base.autoneg = adapter->autoneg;

	return 0;
}
#else /* !ETHTOOL_GLINKSETTINGS */
static __u32 txgbe_backplane_type(struct txgbe_hw *hw)
{
	__u32 mode = 0x00;
	switch (hw->phy.link_mode) {
	case TXGBE_PHYSICAL_LAYER_10GBASE_KX4:
		mode = SUPPORTED_10000baseKX4_Full;
		break;
	case TXGBE_PHYSICAL_LAYER_10GBASE_KR:
		mode = SUPPORTED_10000baseKR_Full;
		break;
	case TXGBE_PHYSICAL_LAYER_1000BASE_KX:
		mode = SUPPORTED_1000baseKX_Full;
		break;
	default:
		mode = (SUPPORTED_10000baseKX4_Full |
			SUPPORTED_10000baseKR_Full |
			SUPPORTED_1000baseKX_Full);
		break;
	}
	return mode;
}

int txgbe_get_settings(struct net_device *netdev,
		       struct ethtool_cmd *ecmd)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 supported_link = 0;
	u32 link_speed = 0;
	bool autoneg = false;
	bool link_up = false;

	TCALL(hw, mac.ops.get_link_capabilities, &supported_link, &autoneg);

	if((hw->subsystem_device_id & 0xF0) == TXGBE_ID_KR_KX_KX4)
		autoneg = adapter->backplane_an ? 1:0;
	else if((hw->subsystem_device_id & 0xF0) == TXGBE_ID_MAC_SGMII)
		autoneg = adapter->autoneg?1:0;

	/* set the supported link speeds */
	if (supported_link & TXGBE_LINK_SPEED_10GB_FULL)
		ecmd->supported |= (txgbe_isbackplane(hw->phy.media_type)) ?
			txgbe_backplane_type(hw) : SUPPORTED_10000baseT_Full;
	if (supported_link & TXGBE_LINK_SPEED_1GB_FULL)
		ecmd->supported |= (txgbe_isbackplane(hw->phy.media_type)) ?
			SUPPORTED_1000baseKX_Full : SUPPORTED_1000baseT_Full;
	if (supported_link & TXGBE_LINK_SPEED_100_FULL)
		ecmd->supported |= SUPPORTED_100baseT_Full;
	if (supported_link & TXGBE_LINK_SPEED_10_FULL)
		ecmd->supported |= SUPPORTED_10baseT_Full;

	/* default advertised speed if phy.autoneg_advertised isn't set */
	ecmd->advertising = ecmd->supported;

	/* set the advertised speeds */
	if (hw->phy.autoneg_advertised) {
		ecmd->advertising = 0;
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_100_FULL)
			ecmd->advertising |= ADVERTISED_100baseT_Full;
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_10GB_FULL)
			ecmd->advertising |= (ecmd->supported & ADVERTISED_MASK_10G);
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_1GB_FULL) {
			if (ecmd->supported & SUPPORTED_1000baseKX_Full)
				ecmd->advertising |= ADVERTISED_1000baseKX_Full;
			else
				ecmd->advertising |= ADVERTISED_1000baseT_Full;
		}
		if (hw->phy.autoneg_advertised & TXGBE_LINK_SPEED_10_FULL)
			ecmd->advertising |= ADVERTISED_10baseT_Full;
	} else {
		/* default modes in case phy.autoneg_advertised isn't set */
		if (supported_link & TXGBE_LINK_SPEED_10GB_FULL)
			ecmd->advertising |= (txgbe_isbackplane(hw->phy.media_type)) ?
			txgbe_backplane_type(hw) : SUPPORTED_10000baseT_Full;
		if (supported_link & TXGBE_LINK_SPEED_1GB_FULL) {
			if (ecmd->supported & SUPPORTED_1000baseKX_Full)
				ecmd->advertising |= ADVERTISED_1000baseKX_Full;
			else
				ecmd->advertising |= ADVERTISED_1000baseT_Full;
		}
		if (supported_link & TXGBE_LINK_SPEED_100_FULL)
			ecmd->advertising |= ADVERTISED_100baseT_Full;
		if (hw->phy.multispeed_fiber && !autoneg) {
			if (supported_link & TXGBE_LINK_SPEED_10GB_FULL)
				ecmd->advertising = ADVERTISED_10000baseT_Full;
		}
		if (supported_link & TXGBE_LINK_SPEED_10_FULL)
			ecmd->advertising |= ADVERTISED_10baseT_Full;
	}

	if (autoneg) {
		ecmd->supported |= SUPPORTED_Autoneg;
		ecmd->advertising |= ADVERTISED_Autoneg;
		ecmd->autoneg = AUTONEG_ENABLE;
	} else
		ecmd->autoneg = AUTONEG_DISABLE;

	ecmd->transceiver = XCVR_EXTERNAL;

	/* Determine the remaining settings based on the PHY type. */
	switch (adapter->hw.phy.type) {
	case txgbe_phy_tn:
	case txgbe_phy_aq:
	case txgbe_phy_cu_unknown:
		ecmd->supported |= SUPPORTED_TP;
		ecmd->advertising |= ADVERTISED_TP;
		ecmd->port = PORT_TP;
		break;
	case txgbe_phy_qt:
		ecmd->supported |= SUPPORTED_FIBRE;
		ecmd->advertising |= ADVERTISED_FIBRE;
		ecmd->port = PORT_FIBRE;
		break;
	case txgbe_phy_nl:
	case txgbe_phy_sfp_passive_tyco:
	case txgbe_phy_sfp_passive_unknown:
	case txgbe_phy_sfp_ftl_active:
	case txgbe_phy_sfp_ftl:
	case txgbe_phy_sfp_avago:
	case txgbe_phy_sfp_intel:
	case txgbe_phy_sfp_unknown:
		switch (adapter->hw.phy.sfp_type) {
			/* SFP+ devices, further checking needed */
		case txgbe_sfp_type_da_cu:
		case txgbe_sfp_type_da_cu_core0:
		case txgbe_sfp_type_da_cu_core1:
		case txgbe_qsfp_type_40g_cu_core0:
		case txgbe_qsfp_type_40g_cu_core1:
			ecmd->supported |= SUPPORTED_FIBRE;
			ecmd->advertising |= ADVERTISED_FIBRE;
			ecmd->port = PORT_DA;
			break;
		case txgbe_sfp_type_sr:
		case txgbe_sfp_type_lr:
		case txgbe_sfp_type_srlr_core0:
		case txgbe_sfp_type_srlr_core1:
		case txgbe_sfp_type_1g_sx_core0:
		case txgbe_sfp_type_1g_sx_core1:
		case txgbe_sfp_type_1g_lx_core0:
		case txgbe_sfp_type_1g_lx_core1:
		case txgbe_sfp_type_da_act_lmt_core0:
		case txgbe_sfp_type_da_act_lmt_core1:
		case txgbe_sfp_type_25g_sr_core0:
		case txgbe_sfp_type_25g_sr_core1:
		case txgbe_sfp_type_25g_lr_core0:
		case txgbe_sfp_type_25g_lr_core1:
		case txgbe_sfp_type_25g_aoc_core0:
		case txgbe_sfp_type_25g_aoc_core1:
		case txgbe_qsfp_type_40g_sr_core0:
		case txgbe_qsfp_type_40g_sr_core1:
		case txgbe_qsfp_type_40g_lr_core0:
		case txgbe_qsfp_type_40g_lr_core1:
		case txgbe_qsfp_type_40g_active_core0:
		case txgbe_qsfp_type_40g_active_core1:
			ecmd->supported |= SUPPORTED_FIBRE;
			ecmd->advertising |= ADVERTISED_FIBRE;
			ecmd->port = PORT_FIBRE;
			break;
		case txgbe_sfp_type_not_present:
			ecmd->supported |= SUPPORTED_FIBRE;
			ecmd->advertising |= ADVERTISED_FIBRE;
			ecmd->port = PORT_NONE;
			break;
		case txgbe_sfp_type_1g_cu_core0:
		case txgbe_sfp_type_1g_cu_core1:
			ecmd->supported |= SUPPORTED_TP;
			ecmd->advertising |= ADVERTISED_TP;
			ecmd->port = PORT_TP;
			break;
		case txgbe_sfp_type_unknown:
		default:
			ecmd->supported |= SUPPORTED_FIBRE;
			ecmd->advertising |= ADVERTISED_FIBRE;
			ecmd->port = PORT_OTHER;
			break;
		}
		break;
	case txgbe_phy_xaui:
		ecmd->supported |= SUPPORTED_TP;
		ecmd->advertising |= ADVERTISED_TP;
		ecmd->port = PORT_TP;
		break;
	case txgbe_phy_unknown:
	case txgbe_phy_generic:
	case txgbe_phy_sfp_unsupported:
	default:
		ecmd->supported |= SUPPORTED_FIBRE;
		ecmd->advertising |= ADVERTISED_FIBRE;
		ecmd->port = PORT_OTHER;
		break;
	}

	if ((hw->subsystem_device_id & 0xF0) == TXGBE_ID_XAUI &&
		(hw->subsystem_device_id & TXGBE_NCSI_MASK) == TXGBE_NCSI_SUP) {
		/* only continue if link was up previously */
		if (!netif_carrier_ok(netdev)) {
			ecmd->speed = -1;
			ecmd->duplex = -1;

			return 0;
		}
	}
	if (!in_interrupt()) {
		TCALL(hw, mac.ops.check_link, &link_speed, &link_up, false);
	} else {
		/*
		 * this case is a special workaround for RHEL5 bonding
		 * that calls this routine from interrupt context
		 */
		link_speed = adapter->link_speed;
		link_up = adapter->link_up;
	}

	ecmd->supported |= SUPPORTED_Pause;

	switch (hw->fc.requested_mode) {
	case txgbe_fc_full:
		ecmd->advertising |= ADVERTISED_Pause;
		break;
	case txgbe_fc_rx_pause:
		ecmd->advertising |= ADVERTISED_Pause |
				     ADVERTISED_Asym_Pause;
		break;
	case txgbe_fc_tx_pause:
		ecmd->advertising |= ADVERTISED_Asym_Pause;
		break;
	default:
		ecmd->advertising &= ~(ADVERTISED_Pause |
				       ADVERTISED_Asym_Pause);
	}

	if (link_up) {
		switch (link_speed) {
		case TXGBE_LINK_SPEED_25GB_FULL:
			ecmd->speed = SPEED_25000;
			break;
		case TXGBE_LINK_SPEED_10GB_FULL:
			ecmd->speed = SPEED_10000;
			break;
		case TXGBE_LINK_SPEED_1GB_FULL:
			ecmd->speed = SPEED_1000;
			break;
		case TXGBE_LINK_SPEED_100_FULL:
			ecmd->speed = SPEED_100;
			break;
		case TXGBE_LINK_SPEED_10_FULL:
			ecmd->speed = SPEED_10;
			break;
		default:
			break;
		}
		ecmd->duplex = DUPLEX_FULL;
	} else {
		ecmd->speed = -1;
		ecmd->duplex = -1;
	}
	if(!adapter->autoneg)
		ecmd->advertising &= ~ADVERTISED_Autoneg;
	ecmd->autoneg = adapter->autoneg?AUTONEG_ENABLE:AUTONEG_DISABLE;
	return 0;
}
#endif /* !ETHTOOL_GLINKSETTINGS */

#ifdef ETHTOOL_GLINKSETTINGS

static int txgbe_set_link_ksettings(struct net_device *netdev,
				const struct ethtool_link_ksettings *cmd)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 advertised, old, link_support;
	bool autoneg;
	s32 err = 0;
	struct ethtool_link_ksettings temp_ks;
	u32 curr_autoneg = 2;
	
	if((hw->subsystem_device_id & 0xF0) == TXGBE_ID_KR_KX_KX4)
		adapter->backplane_an = cmd->base.autoneg ? 1 : 0;
	if ((hw->subsystem_device_id & 0xF0) == TXGBE_ID_MAC_SGMII)
		adapter->autoneg = cmd->base.autoneg ? 1 : 0;
	if ((hw->phy.media_type == txgbe_media_type_copper) || (hw->phy.multispeed_fiber)) {
		memcpy(&temp_ks, cmd, sizeof(struct ethtool_link_ksettings));
		/* To be compatible with test cases */
		if (hw->phy.media_type == txgbe_media_type_fiber) {
			if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  25000baseSR_Full)) {
				ethtool_link_ksettings_add_link_mode(&temp_ks, supported,
								     25000baseSR_Full);
			}

			if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  10000baseT_Full)) {
				ethtool_link_ksettings_add_link_mode(&temp_ks, supported,
								     10000baseT_Full);
#ifndef HAVE_NO_ETHTOOL_10000SR
				ethtool_link_ksettings_del_link_mode(&temp_ks, supported,
								     10000baseSR_Full);
#endif
#ifndef HAVE_NO_ETHTOOL_10000LR
				ethtool_link_ksettings_del_link_mode(&temp_ks, supported,
								     10000baseLR_Full);
#endif
			}
			if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  1000baseT_Full)) {
				ethtool_link_ksettings_add_link_mode(&temp_ks, supported,
								     1000baseT_Full);
#ifndef HAVE_NO_ETHTOOL_1000X
				ethtool_link_ksettings_del_link_mode(&temp_ks, supported,
								     1000baseX_Full);
#endif
			}
		}

		/*
		 * this function does not support duplex forcing, but can
		 * limit the advertising of the adapter to the specified speed
		 */
		if (!bitmap_subset(cmd->link_modes.advertising, temp_ks.link_modes.supported,
				   __ETHTOOL_LINK_MODE_MASK_NBITS))
			return -EINVAL;

		old = hw->phy.autoneg_advertised;
		advertised = 0;
		if (!cmd->base.autoneg) {
			if (cmd->base.speed == SPEED_25000)
				advertised |= TXGBE_LINK_SPEED_25GB_FULL;
			else if (cmd->base.speed == SPEED_10000)
				advertised |= TXGBE_LINK_SPEED_10GB_FULL;
			else if (cmd->base.speed == SPEED_1000)
				advertised |= TXGBE_LINK_SPEED_1GB_FULL;
			else
				advertised |= old;
		}else{
			if (ethtool_link_ksettings_test_link_mode(cmd, advertising, 25000baseSR_Full))
				advertised |= TXGBE_LINK_SPEED_25GB_FULL;

			if (ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseSR_Full) ||
			ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseLR_Full) ||
			ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseT_Full))
				advertised |= TXGBE_LINK_SPEED_10GB_FULL;

			if (ethtool_link_ksettings_test_link_mode(cmd, advertising, 1000baseX_Full) ||
			ethtool_link_ksettings_test_link_mode(cmd, advertising, 1000baseT_Full))
				advertised |= TXGBE_LINK_SPEED_1GB_FULL;

			if (ethtool_link_ksettings_test_link_mode(cmd, advertising, 100baseT_Full))
				advertised |= TXGBE_LINK_SPEED_100_FULL;

			if (ethtool_link_ksettings_test_link_mode(cmd, advertising, 10baseT_Full))
				advertised |= TXGBE_LINK_SPEED_10_FULL;
		}

		if (advertised == TXGBE_LINK_SPEED_1GB_FULL &&
		    hw->phy.media_type != txgbe_media_type_copper) {
			curr_autoneg = txgbe_rd32_epcs(hw, TXGBE_SR_MII_MMD_CTL);
			curr_autoneg = !!(curr_autoneg & (0x1 << 12));
			if (old == advertised && (curr_autoneg == !!(cmd->base.autoneg)))
				return 0;
		}

		err = TCALL(hw, mac.ops.get_link_capabilities,
			&link_support, &autoneg);
		if (err)
			e_info(probe, "get link capabiliyies failed with code %d\n", err);
		if (!(link_support & advertised)) {
			e_info(probe, "unsupported advertised: %x", advertised);
			return -EINVAL;
		}

		/* this sets the link speed and restarts auto-neg */
		while (test_and_set_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
			usleep_range(1000, 2000);
		adapter->autoneg = cmd->base.autoneg ? 1 : 0;
		hw->mac.autotry_restart = true;
		adapter->flags |= TXGBE_FLAG_NEED_LINK_UPDATE;
		txgbe_service_event_schedule(adapter);
		err = TCALL(hw, mac.ops.setup_link, advertised, true);
		if (err) {
			e_info(probe, "setup link failed with code %d\n", err);
			TCALL(hw, mac.ops.setup_link, old, true);
		}
		if ((hw->subsystem_device_id & TXGBE_NCSI_MASK) == TXGBE_NCSI_SUP)
				TCALL(hw, mac.ops.flap_tx_laser);

		/* notify fw autoneg status */
		txgbe_hic_write_autoneg_status(hw, cmd->base.autoneg);

		clear_bit(__TXGBE_IN_SFP_INIT, &adapter->state);
	} else if ((hw->subsystem_device_id & 0xF0) == TXGBE_ID_KR_KX_KX4 ||
		   (hw->subsystem_device_id & 0xF0) == TXGBE_ID_MAC_SGMII) {
		if (!cmd->base.autoneg) {
			if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  10000baseKR_Full) &
			    ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  1000baseKX_Full) &
			    ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  10000baseKX4_Full))
				return -EINVAL;
		}

		advertised = 0;
		if (ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseKR_Full)) {
			err = txgbe_set_link_to_kr(hw, 1);
			advertised |= TXGBE_LINK_SPEED_10GB_FULL;
		} else if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								 10000baseKX4_Full)) {
			err = txgbe_set_link_to_kx4(hw, 1);
			advertised |= TXGBE_LINK_SPEED_10GB_FULL;
		} else if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								 1000baseKX_Full)) {
			advertised |= TXGBE_LINK_SPEED_1GB_FULL;
			err = txgbe_set_link_to_kx(hw, TXGBE_LINK_SPEED_1GB_FULL, 0);
			txgbe_set_sgmii_an37_ability(hw);
		}
		if (err)
			err = -EACCES;

		return err;
	} else {
		/* in this case we currently only support 10Gb/FULL */
		u32 speed = cmd->base.speed;
		if (hw->mac.type == txgbe_mac_aml || hw->mac.type == txgbe_mac_aml40) {
			return -EINVAL;
		} else if ((ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseT_Full) ||
		     ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseKR_Full) ||
		     ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseKX4_Full) ||
		     ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseLR_Full) ||
			 ethtool_link_ksettings_test_link_mode(cmd, advertising, 10000baseSR_Full))) {
			if ((cmd->base.autoneg == AUTONEG_ENABLE) ||
			    (!ethtool_link_ksettings_test_link_mode(cmd, advertising,
								    10000baseT_Full)) ||
			    (speed + cmd->base.duplex != SPEED_10000 + DUPLEX_FULL))
				return -EINVAL;
		} else if ((ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  1000baseT_Full) ||
			    ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  1000baseKX_Full) ||
			    ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  1000baseX_Full))) {
			memcpy(&temp_ks, cmd, sizeof(struct ethtool_link_ksettings));

			if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  1000baseT_Full)) {
				ethtool_link_ksettings_add_link_mode(&temp_ks, supported,
								     1000baseT_Full);
#ifndef HAVE_NO_ETHTOOL_1000X
				ethtool_link_ksettings_del_link_mode(&temp_ks, supported,
								     1000baseX_Full);
#endif
				ethtool_link_ksettings_del_link_mode(&temp_ks, supported,
								     1000baseKX_Full);
			}

			if (!bitmap_subset(cmd->link_modes.advertising,
					   temp_ks.link_modes.supported,
					   __ETHTOOL_LINK_MODE_MASK_NBITS))
				return -EINVAL;

			old = hw->phy.autoneg_advertised;
			advertised = 0;

			if (ethtool_link_ksettings_test_link_mode(cmd, advertising,
								  1000baseX_Full) ||
			    ethtool_link_ksettings_test_link_mode(cmd, advertising, 1000baseT_Full))
				advertised |= TXGBE_LINK_SPEED_1GB_FULL;


#if 0
			if (hw->mac.type == txgbe_mac_aml) {
				curr_autoneg = txgbe_rd32_epcs(hw, SR_AN_CTRL);
				curr_autoneg = !!(curr_autoneg & (0x1 << 12));
			}
#endif
			if (advertised == TXGBE_LINK_SPEED_1GB_FULL) {
				curr_autoneg = txgbe_rd32_epcs(hw, TXGBE_SR_MII_MMD_CTL);
				curr_autoneg = !!(curr_autoneg & (0x1 << 12));
			}
			if (old == advertised && (curr_autoneg == !!cmd->base.autoneg))
				return -EINVAL;
			/* this sets the link speed and restarts auto-neg */
			while (test_and_set_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
				usleep_range(1000, 2000);

			adapter->autoneg = cmd->base.autoneg?1:0;
			hw->mac.autotry_restart = true;
			err = TCALL(hw, mac.ops.setup_link, advertised, true);
			if (err) {
				e_info(probe, "setup link failed with code %d\n", err);
				TCALL(hw, mac.ops.setup_link, old, true);
			}
			if ((hw->subsystem_device_id & TXGBE_NCSI_MASK) == TXGBE_NCSI_SUP)
				TCALL(hw, mac.ops.flap_tx_laser);

			/* notify fw autoneg status */
			txgbe_hic_write_autoneg_status(hw, cmd->base.autoneg);

			clear_bit(__TXGBE_IN_SFP_INIT, &adapter->state);
		}
		adapter->autoneg = cmd->base.autoneg?1:0;
	}
	if (err)
		return -EINVAL;

	return err;
}
#else /* !ETHTOOL_GLINKSETTINGS */
static int txgbe_set_settings(struct net_device *netdev,
			      struct ethtool_cmd *ecmd)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 advertised, old;
	u32 curr_autoneg = 2;
	s32 err = 0;

	if((hw->subsystem_device_id & 0xF0) == TXGBE_ID_KR_KX_KX4)
		adapter->backplane_an = ecmd->autoneg ? 1 : 0;
	if ((hw->subsystem_device_id & 0xF0) == TXGBE_ID_MAC_SGMII)
		adapter->autoneg = ecmd->autoneg? 1 : 0;

	if ((hw->phy.media_type == txgbe_media_type_copper) ||
	    (hw->phy.multispeed_fiber)) {
		/*
		 * this function does not support duplex forcing, but can
		 * limit the advertising of the adapter to the specified speed
		 */
		if (ecmd->advertising & ~ecmd->supported)
			return -EINVAL;

		/* only allow one speed at a time if no autoneg */
		if (!ecmd->autoneg && hw->phy.multispeed_fiber) {
			if (ecmd->advertising ==
			    (ADVERTISED_10000baseT_Full | ADVERTISED_1000baseT_Full))
				return -EINVAL;
		}

		old = hw->phy.autoneg_advertised;
		advertised = 0;

		if (ecmd->advertising & ADVERTISED_10000baseT_Full)
			advertised |= TXGBE_LINK_SPEED_10GB_FULL;

		if (ecmd->advertising & ADVERTISED_1000baseT_Full)
			advertised |= TXGBE_LINK_SPEED_1GB_FULL;

		if (ecmd->advertising & ADVERTISED_100baseT_Full)
			advertised |= TXGBE_LINK_SPEED_100_FULL;

		if (ecmd->advertising & ADVERTISED_10baseT_Full)
			advertised |= TXGBE_LINK_SPEED_10_FULL;

		if (advertised == TXGBE_LINK_SPEED_1GB_FULL &&
		    hw->phy.media_type != txgbe_media_type_copper) {
			curr_autoneg = txgbe_rd32_epcs(hw, TXGBE_SR_MII_MMD_CTL);
			curr_autoneg = !!(curr_autoneg & (0x1 << 12));
			if (old == advertised && (curr_autoneg == !!ecmd->autoneg))
				return err;
		}

		if (advertised == TXGBE_LINK_SPEED_10GB_FULL &&
			ecmd->autoneg == AUTONEG_DISABLE)
			return -EINVAL;

		/* this sets the link speed and restarts auto-neg */
		while (test_and_set_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
			usleep_range(1000, 2000);

		adapter->autoneg = ecmd->autoneg ? 1 : 0;
		hw->mac.autotry_restart = true;
		err = TCALL(hw, mac.ops.setup_link, advertised, true);
		if (err) {
			e_info(probe, "setup link failed with code %d\n", err);
			TCALL(hw, mac.ops.setup_link, old, true);
		}

		if ((hw->subsystem_device_id & TXGBE_NCSI_MASK) == TXGBE_NCSI_SUP)
				TCALL(hw, mac.ops.flap_tx_laser);
		
		/* notify fw autoneg status */
		txgbe_hic_write_autoneg_status(hw, ecmd->autoneg);

		clear_bit(__TXGBE_IN_SFP_INIT, &adapter->state);
	} else if ((hw->subsystem_device_id & 0xF0) == TXGBE_ID_KR_KX_KX4 ||
		   (hw->subsystem_device_id & 0xF0) == TXGBE_ID_MAC_SGMII) {
		if (!ecmd->autoneg) {
			if (ecmd->advertising ==
			    (ADVERTISED_10000baseKR_Full | ADVERTISED_1000baseKX_Full |
			     ADVERTISED_10000baseKX4_Full))
				return -EINVAL;
		}

		advertised = 0;
		if (ecmd->advertising & ADVERTISED_10000baseKR_Full) {
			err = txgbe_set_link_to_kr(hw, 1);
			advertised |= TXGBE_LINK_SPEED_10GB_FULL;
		} else if (ecmd->advertising & ADVERTISED_10000baseKX4_Full) {
			err = txgbe_set_link_to_kx4(hw, 1);
			advertised |= TXGBE_LINK_SPEED_10GB_FULL;
		} else if (ecmd->advertising & ADVERTISED_1000baseKX_Full) {
			advertised |= TXGBE_LINK_SPEED_1GB_FULL;
			err = txgbe_set_link_to_kx(hw, TXGBE_LINK_SPEED_1GB_FULL, 0);
			txgbe_set_sgmii_an37_ability(hw);
		}
		if (err)
			return -EACCES;
		return err;
	} else {
		/* in this case we currently only support 10Gb/FULL and 1Gb/FULL*/
		if (hw->mac.type == txgbe_mac_aml || hw->mac.type == txgbe_mac_aml40) {
			return -EINVAL;
		} else if (ecmd->advertising & ADVERTISED_1000baseT_Full) {
			if (ecmd->advertising & ~ecmd->supported)
				return -EINVAL;

			old = hw->phy.autoneg_advertised;
			advertised = 0;

			if (ecmd->advertising & ADVERTISED_1000baseT_Full)
				advertised |= TXGBE_LINK_SPEED_1GB_FULL;

			if (advertised == TXGBE_LINK_SPEED_1GB_FULL) {
				curr_autoneg = txgbe_rd32_epcs(hw, TXGBE_SR_MII_MMD_CTL);
				curr_autoneg = !!(curr_autoneg & (0x1 << 12));
			}
			if (old == advertised && (curr_autoneg == !!ecmd->autoneg))
				return err;
			/* this sets the link speed and restarts auto-neg */
			while (test_and_set_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
				usleep_range(1000, 2000);

			adapter->autoneg = ecmd->autoneg ? 1 : 0;
			hw->mac.autotry_restart = true;
			err = TCALL(hw, mac.ops.setup_link, advertised, true);
			if (err) {
				e_info(probe, "setup link failed with code %d\n", err);
				TCALL(hw, mac.ops.setup_link, old, true);
			}
			if ((hw->subsystem_device_id & TXGBE_NCSI_MASK) == TXGBE_NCSI_SUP)
				TCALL(hw, mac.ops.flap_tx_laser);
			
			/* notify fw autoneg status */
			txgbe_hic_write_autoneg_status(hw, ecmd->autoneg);

			clear_bit(__TXGBE_IN_SFP_INIT, &adapter->state);
		}
		adapter->autoneg = ecmd->autoneg ? 1 : 0;
	}

	if (err)
		return -EINVAL;

	return err;
}
#endif /* !ETHTOOL_GLINKSETTINGS */
#ifdef ETHTOOL_GFECPARAM
static int txgbe_get_fec_param(struct net_device *netdev,
			      struct ethtool_fecparam *fecparam)
{
	int err = 0;
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 supported_link = 0;
	bool autoneg = false;
	u32 speed = 0;
	bool link_up;

	TCALL(hw, mac.ops.get_link_capabilities, &supported_link, &autoneg);

	if (hw->mac.type != txgbe_mac_aml) {
		err = -EAGAIN;
		goto done;
	}
	TCALL(hw, mac.ops.check_link, &speed, &link_up, false);
	fecparam->fec = 0;
	if (supported_link == TXGBE_LINK_SPEED_10GB_FULL) {
		fecparam->fec |= ETHTOOL_FEC_OFF;
	} else {
		if (adapter->fec_link_mode == TXGBE_PHY_FEC_AUTO)
			fecparam->fec |= ETHTOOL_FEC_AUTO;
		else if (adapter->fec_link_mode & TXGBE_PHY_FEC_BASER)
			fecparam->fec |= ETHTOOL_FEC_BASER;
		else if (adapter->fec_link_mode & TXGBE_PHY_FEC_RS)
			fecparam->fec |= ETHTOOL_FEC_RS;
		else
			fecparam->fec |= ETHTOOL_FEC_OFF;
	}
	if (speed == TXGBE_LINK_SPEED_10GB_FULL) {
		fecparam->active_fec = ETHTOOL_FEC_OFF;
		goto done;
	}
	if (!link_up) {
		fecparam->active_fec = ETHTOOL_FEC_OFF;
		goto done;
	}
	switch (adapter->cur_fec_link) {
	case TXGBE_PHY_FEC_BASER:
		fecparam->active_fec = ETHTOOL_FEC_BASER;
		break;
	case TXGBE_PHY_FEC_RS:
		fecparam->active_fec = ETHTOOL_FEC_RS;
		break;
	case TXGBE_PHY_FEC_OFF:
		fecparam->active_fec = ETHTOOL_FEC_OFF;
		break;
	default:
		fecparam->active_fec = ETHTOOL_FEC_OFF;
		break;
	}
done:
	return err;
}

static int txgbe_set_fec_param(struct net_device *netdev,
			      struct ethtool_fecparam *fecparam)
{
	int err = 0;
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u8 cur_fec_mode = adapter->fec_link_mode;
	bool autoneg = false;
	u32 supported_link = 0;

	TCALL(hw, mac.ops.get_link_capabilities, &supported_link, &autoneg);

	if (hw->mac.type != txgbe_mac_aml) {
		err = -EAGAIN;
		goto done;
	}

	switch (fecparam->fec) {
	case ETHTOOL_FEC_AUTO:
		adapter->fec_link_mode = TXGBE_PHY_FEC_AUTO;
		break;
	case ETHTOOL_FEC_BASER:
		adapter->fec_link_mode = TXGBE_PHY_FEC_BASER;
		break;
	case ETHTOOL_FEC_OFF:
	case ETHTOOL_FEC_NONE:
		adapter->fec_link_mode = TXGBE_PHY_FEC_OFF;
		break;
	case ETHTOOL_FEC_RS:
		adapter->fec_link_mode = TXGBE_PHY_FEC_RS;
		break;
	default:
		e_warn(drv, "Unsupported FEC mode: %d",
			 fecparam->fec);
		err = -EINVAL;
		goto done;
	}
	if (cur_fec_mode != adapter->fec_link_mode) {
		/* reset link */
		int status, i;
		u32 link_speed = TXGBE_LINK_SPEED_UNKNOWN;
		bool link_up = false;
		if (hw->phy.multispeed_fiber && (hw->phy.autoneg_advertised == (TXGBE_LINK_SPEED_10GB_FULL | TXGBE_LINK_SPEED_25GB_FULL)) &&
		    adapter->fec_link_mode != TXGBE_PHY_FEC_AUTO)
		{
			while (test_and_set_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
				usleep_range(1000, 2000);
			status = TCALL(hw, mac.ops.setup_mac_link,
				       TXGBE_LINK_SPEED_25GB_FULL, 0);
			if (status != 0)
				return status;
			for (i = 0; i < 30; i++) {
				txgbe_e56_check_phy_link(hw, &link_speed, &link_up);
				if (link_up){
					goto out;}
				msleep(200);
			}
			msec_delay(100);
			status = TCALL(hw, mac.ops.setup_mac_link,
				TXGBE_LINK_SPEED_10GB_FULL, 0);
			if (status != 0)
				return status;
			for (i = 0; i < 35;i++) {
				u32 link_speed;
				bool link_up = 0;
				txgbe_e56_check_phy_link(hw, &link_speed, &link_up);
				if (link_up) {
					if (rd32(hw, 0x14404) & 0x1) {
						adapter->flags &= ~TXGBE_FLAG_NEED_LINK_CONFIG;
						mutex_lock(&adapter->e56_lock);
						txgbe_wr32_ephy(hw, E56PHY_INTR_1_ADDR, E56PHY_INTR_1_IDLE_EXIT1);
						mutex_unlock(&adapter->e56_lock);
						goto out;
					}
				}
				msleep(200);
			}
			adapter->flags |= TXGBE_FLAG_NEED_LINK_CONFIG;
			txgbe_service_event_schedule(adapter);
out:
			clear_bit(__TXGBE_IN_SFP_INIT, &adapter->state);
		} else if (hw->phy.multispeed_fiber) {
			while (test_and_set_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
				usleep_range(1000, 2000);
			TCALL(hw, mac.ops.setup_link, hw->phy.autoneg_advertised, true);
			clear_bit(__TXGBE_IN_SFP_INIT, &adapter->state);
		} else {
			adapter->flags |= TXGBE_FLAG_NEED_LINK_CONFIG;
			txgbe_service_event_schedule(adapter);
		}
	}
done:
	return err;
}
#endif /* ETHTOOL_GFECPARAM */
static void txgbe_get_pauseparam(struct net_device *netdev,
				 struct ethtool_pauseparam *pause)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;

	if (txgbe_device_supports_autoneg_fc(hw) &&
	    !hw->fc.disable_fc_autoneg)
		pause->autoneg = 1;
	else
		pause->autoneg = 0;

	if (hw->fc.current_mode == txgbe_fc_rx_pause) {
		pause->rx_pause = 1;
	} else if (hw->fc.current_mode == txgbe_fc_tx_pause) {
		pause->tx_pause = 1;
	} else if (hw->fc.current_mode == txgbe_fc_full) {
		pause->rx_pause = 1;
		pause->tx_pause = 1;
	}
}

static int txgbe_set_pauseparam(struct net_device *netdev,
				struct ethtool_pauseparam *pause)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	struct txgbe_fc_info fc = hw->fc;


	/* some devices do not support autoneg of flow control */
	if ((pause->autoneg == AUTONEG_ENABLE) &&
	    !txgbe_device_supports_autoneg_fc(hw))
	    return -EINVAL;

	fc.disable_fc_autoneg = (pause->autoneg != AUTONEG_ENABLE);

	if ((pause->rx_pause && pause->tx_pause) || pause->autoneg)
		fc.requested_mode = txgbe_fc_full;
	else if (pause->rx_pause)
		fc.requested_mode = txgbe_fc_rx_pause;
	else if (pause->tx_pause)
		fc.requested_mode = txgbe_fc_tx_pause;
	else
		fc.requested_mode = txgbe_fc_none;

	/* if the thing changed then we'll update and use new autoneg */
	if (memcmp(&fc, &hw->fc, sizeof(struct txgbe_fc_info))) {
		hw->fc = fc;
		if (netif_running(netdev))
			txgbe_reinit_locked(adapter);
		else
			txgbe_reset(adapter);
	}

	return 0;
}

static u32 txgbe_get_msglevel(struct net_device *netdev)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	return adapter->msg_enable;
}

static void txgbe_set_msglevel(struct net_device *netdev, u32 data)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	adapter->msg_enable = data;
}

static int txgbe_get_regs_len(struct net_device __always_unused *netdev)
{
#define TXGBE_REGS_LEN  4096
	return TXGBE_REGS_LEN * sizeof(u32);
}

#define TXGBE_GET_STAT(_A_, _R_)        (_A_->stats._R_)


static void txgbe_get_regs(struct net_device *netdev, struct ethtool_regs *regs,
			   void *p)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 *regs_buff = p;
	u32 i;
	u32 id = 0;

	memset(p, 0, TXGBE_REGS_LEN * sizeof(u32));
	regs_buff[TXGBE_REGS_LEN - 1] = 0x55555555;

	regs->version = hw->revision_id << 16 |
			hw->device_id;

	/* Global Registers */
	/* chip control */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_MIS_PWR);//0
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_MIS_CTL);//1
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_MIS_PF_SM);//2
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_MIS_RST);//3
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_MIS_ST);//4
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_MIS_SWSM);//5
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_MIS_RST_ST);//6
	/* pvt sensor */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TS_CTL);//7
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TS_EN);//8
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TS_ST);//9
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TS_ALARM_THRE);//10
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TS_DALARM_THRE);//11
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TS_INT_EN);//12
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TS_ALARM_ST);//13
	/* Fmgr Register */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_CMD);//14
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_DATA);//15
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_STATUS);//16
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_USR_CMD);//17
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_CMDCFG0);//18
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_CMDCFG1);//19
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_ILDR_STATUS);//20
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_SPI_ILDR_SWPTR);//21

	/* Port Registers */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_PORT_CTL);//22
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_PORT_ST);//23
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_EX_VTYPE);//24
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_VXLAN);//25
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_VXLAN_GPE);//26
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_GENEVE);//27
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_TEREDO);//28
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_TCP_TIME);//29
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_LED_CTL);//30
	/* GPIO */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_GPIO_DR);//31
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_GPIO_DDR);//32
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_GPIO_CTL);//33
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_GPIO_INTEN);//34
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_GPIO_INTMASK);//35
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_GPIO_INTSTATUS);//36
	/* I2C */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CON);//37
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_TAR);//38
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_DATA_CMD);//39
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_SS_SCL_HCNT);//40
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_SS_SCL_LCNT);//41
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_FS_SCL_HCNT);//42
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_FS_SCL_LCNT);//43
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_HS_SCL_HCNT);//44
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_INTR_STAT);//45
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_INTR_MASK);//46
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_RAW_INTR_STAT);//47
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_RX_TL);//48
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_TX_TL);//49
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_INTR);//50
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_RX_UNDER);//51
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_RX_OVER);//52
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_TX_OVER);//53
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_RD_REQ);//54
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_TX_ABRT);//55
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_RX_DONE);//56
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_ACTIVITY);//57
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_STOP_DET);//58
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_START_DET);//59
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_GEN_CALL);//60
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_ENABLE);//61
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_STATUS);//62
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_TXFLR);//63
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_RXFLR);//64
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_SDA_HOLD);//65
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_TX_ABRT_SOURCE);//66
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_SDA_SETUP);//67
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_ENABLE_STATUS);//68
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_FS_SPKLEN);//69
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_HS_SPKLEN);//70
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_SCL_STUCK_TIMEOUT);//71
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_SDA_STUCK_TIMEOUT);//72
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_CLR_SCL_STUCK_DET);//73
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_DEVICE_ID);//74
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_COMP_PARAM_1);//75
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_COMP_VERSION);//76
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_I2C_COMP_TYPE);//77
	/* TX TPH */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_TPH_TDESC);//78
	/* RX TPH */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_TPH_RDESC);//79
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_TPH_RHDR);//80
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_CFG_TPH_RPL);//81

	/* TDMA */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_CTL);//82
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_VF_TE(0));//83
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_VF_TE(1));//84
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_PB_THRE(i));//85-92
	}
	for (i = 0; i < 4; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_LLQ(i));//93-96
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_ETYPE_LB_L);//97
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_ETYPE_LB_H);//98
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_ETYPE_AS_L);//99
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_ETYPE_AS_H);//100
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_MAC_AS_L);//101
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_MAC_AS_H);//102
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_VLAN_AS_L);//103
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_VLAN_AS_H);//104
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_TCP_FLG_L);//105
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_TCP_FLG_H);//106
	for (i = 0; i < 64; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_VLAN_INS(i));//107-234
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_ETAG_INS(i));
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_PBWARB_CTL);//235
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_MMW);//236
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_PBWARB_CFG(i));//237-244
	}
	for (i = 0; i < 128; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_VM_CREDIT(i));//245-372
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_FC_EOF);//373
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDM_FC_SOF);//374

	/* RDMA */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDM_ARB_CTL);//375
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDM_VF_RE(0));//376
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDM_VF_RE(1));//377
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDM_RSC_CTL);//378
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDM_ARB_CFG(i));//379-386
	}
	for (i = 0; i < 4; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDM_PF_QDE(i));//387-394
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDM_PF_HIDE(i));
	}

	/* RDB */
	/*flow control */
	for (i = 0; i < 4; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RFCV(i));//395-398
	}
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RFCL(i));//399-414
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RFCH(i));
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RFCRT);//415
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RFCC);//416
	/* receive packet buffer */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_PB_CTL);//417
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_PB_WRAP);//418
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_UP2TC);//419
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_PB_SZ(i));//420-435
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_MPCNT(i));
	}
	/* lli interrupt */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_LLI_THRE);//436
	/* ring assignment */
	for (i = 0; i < 64; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_PL_CFG(i));//437-500
	}
	for (i = 0; i < 32; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RSSTBL(i));//501-532
	}
	for (i = 0; i < 10; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RSSRK(i));//533-542
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RSS_TC);//543
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_RA_CTL);//544
	for (i = 0; i < 128; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_5T_SA(i));//545-1184
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_5T_DA(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_5T_SDP(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_5T_CTL0(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_5T_CTL1(i));
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_SYN_CLS);//1185
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_ETYPE_CLS(i));//1186-1193
	}
	/* fcoe redirection table */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FCRE_CTL);//1194
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FCRE_TBL(i));//1195-1202
	}
	/*flow director */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_CTL);//1203
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_HKEY);//1204
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_SKEY);//1205
	for (i = 0; i < 16; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_FLEX_CFG(i));//1206-1221
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_FREE);//1222
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_LEN);//1223
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_USE_ST);//1224
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_FAIL_ST);//1225
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_MATCH);//1226
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_MISS);//1227
	for (i = 0; i < 3; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_IP6(i));//1228-1230
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_SA);//1231
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_DA);//1232
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_PORT);//1233
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_FLEX);//1234
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_HASH);//1235
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_CMD);//1236
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_DA4_MSK);//1237
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_SA4_MSK);//1238
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_TCP_MSK);//1239
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_UDP_MSK);//1240
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_SCTP_MSK);//1241
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_IP6_MSK);//1242
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RDB_FDIR_OTHER_MSK);//1243

	/* PSR */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_CTL);//1244
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VLAN_CTL);//1245
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VM_CTL);//1246
	for (i = 0; i < 64; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VM_L2CTL(i));//1247-1310
	}
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_ETYPE_SWC(i));//1311-1318
	}
	for (i = 0; i < 128; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MC_TBL(i));//1319-1702
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_UC_TBL(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VLAN_TBL(i));
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MAC_SWC_AD_L);//1703
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MAC_SWC_AD_H);//1704
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MAC_SWC_VM_L);//1705
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MAC_SWC_VM_H);//1706
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MAC_SWC_IDX);//1707
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VLAN_SWC);//1708
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VLAN_SWC_VM_L);//1709
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VLAN_SWC_VM_H);//1710
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_VLAN_SWC_IDX);//1711
	for (i = 0; i < 4; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MR_CTL(i));//1712-1731
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MR_VLAN_L(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MR_VLAN_H(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MR_VM_L(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_MR_VM_H(i));
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_1588_CTL);//1732
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_1588_STMPL);//1733
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_1588_STMPH);//1734
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_1588_ATTRL);//1735
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_1588_ATTRH);//1736
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_1588_MSGTYPE);//1737
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_WKUP_CTL);//1738
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_WKUP_IPV);//1739
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_LAN_FLEX_CTL);//1740
	for (i = 0; i < 4; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_WKUP_IP4TBL(i));//1741-1748
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_WKUP_IP6TBL(i));
	}
	for (i = 0; i < 16; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_LAN_FLEX_DW_L(i));//1749-1796
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_LAN_FLEX_DW_H(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_LAN_FLEX_MSK(i));
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PSR_LAN_FLEX_CTL);//1797

	/* TDB */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDB_TFCS);//1798
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDB_PB_SZ(0));//1799
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDB_UP2TC);//1800
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDB_PBRARB_CTL);//1801
	for (i = 0; i < 8; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDB_PBRARB_CFG(i));//1802-1809
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TDB_MNG_TC);//1810

	/* tsec */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_CTL);//1811
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_ST);//1812
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_BUF_AF);//1813
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_BUF_AE);//1814
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_MIN_IFG);//1815
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_CTL);//1816
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_STMPL);//1817
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_STMPH);//1818
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_SYSTIML);//1819
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_SYSTIMH);//1820
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_INC);//1821
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_ADJL);//1822
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_TSC_1588_ADJH);//1823

	/* RSEC */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RSC_CTL);//1824
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_RSC_ST);//1825

	/* BAR register */
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_MISC_IC);//1826
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_MISC_ICS);//1827
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_MISC_IEN);//1828
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_GPIE);//1829
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_IC(0));//1830
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_IC(1));//1831
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_ICS(0));//1832
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_ICS(1));//1833
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_IMS(0));//1834
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_IMS(1));//1835
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_IMC(0));//1836
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_IMC(1));//1837
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_ISB_ADDR_L);//1838
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_ISB_ADDR_H);//1839
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_ITRSEL);//1840
	for (i = 0; i < 64; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_ITR(i));//1841-1968
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_IVAR(i));
	}
	regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_MISC_IVAR);//1969
	for (i = 0; i < 128; i++) {
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_RR_BAL(i));//1970-3249
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_RR_BAH(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_RR_WP(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_RR_RP(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_RR_CFG(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_TR_BAL(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_TR_BAH(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_TR_WP(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_TR_RP(i));
		regs_buff[id++] = TXGBE_R32_Q(hw, TXGBE_PX_TR_CFG(i));
	}
}

static int txgbe_get_eeprom_len(struct net_device *netdev)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	return adapter->hw.eeprom.word_size * 2;
}

static int txgbe_get_eeprom(struct net_device *netdev,
			    struct ethtool_eeprom *eeprom, u8 *bytes)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u16 *eeprom_buff;
	int first_word, last_word, eeprom_len;
	int ret_val = 0;
	u16 i;

	if (eeprom->len == 0)
		return -EINVAL;

	eeprom->magic = hw->vendor_id | (hw->device_id << 16);

	first_word = eeprom->offset >> 1;
	last_word = (eeprom->offset + eeprom->len - 1) >> 1;
	eeprom_len = last_word - first_word + 1;

	eeprom_buff = kmalloc(sizeof(u16) * eeprom_len, GFP_KERNEL);
	if (!eeprom_buff)
		return -ENOMEM;

	ret_val = TCALL(hw, eeprom.ops.read_buffer, first_word, eeprom_len,
					   eeprom_buff);

	/* Device's eeprom is always little-endian, word addressable */
	for (i = 0; i < eeprom_len; i++)
		le16_to_cpus(&eeprom_buff[i]);

	memcpy(bytes, (u8 *)eeprom_buff + (eeprom->offset & 1), eeprom->len);
	kfree(eeprom_buff);

	return ret_val;
}

static int txgbe_set_eeprom(struct net_device *netdev,
			    struct ethtool_eeprom *eeprom, u8 *bytes)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u16 *eeprom_buff;
	void *ptr;
	int max_len, first_word, last_word, ret_val = 0;
	u16 i;

	if (eeprom->len == 0)
		return -EINVAL;

	if (eeprom->magic != (hw->vendor_id | (hw->device_id << 16)))
		return -EINVAL;

	max_len = hw->eeprom.word_size * 2;

	first_word = eeprom->offset >> 1;
	last_word = (eeprom->offset + eeprom->len - 1) >> 1;
	eeprom_buff = kmalloc(max_len, GFP_KERNEL);
	if (!eeprom_buff)
		return -ENOMEM;

	ptr = eeprom_buff;

	if (eeprom->offset & 1) {
		/*
		 * need read/modify/write of first changed EEPROM word
		 * only the second byte of the word is being modified
		 */
		ret_val = TCALL(hw, eeprom.ops.read, first_word,
				&eeprom_buff[0]);
		if (ret_val)
			goto err;

		ptr++;
	}
	if (((eeprom->offset + eeprom->len) & 1) && (ret_val == 0)) {
		/*
		 * need read/modify/write of last changed EEPROM word
		 * only the first byte of the word is being modified
		 */
		ret_val = TCALL(hw, eeprom.ops.read, last_word,
				&eeprom_buff[last_word - first_word]);
		if (ret_val)
			goto err;
	}

	/* Device's eeprom is always little-endian, word addressable */
	for (i = 0; i < last_word - first_word + 1; i++)
		le16_to_cpus(&eeprom_buff[i]);

	memcpy(ptr, bytes, eeprom->len);

	for (i = 0; i < last_word - first_word + 1; i++)
		cpu_to_le16s(&eeprom_buff[i]);

	ret_val = TCALL(hw, eeprom.ops.write_buffer, first_word,
					    last_word - first_word + 1,
					    eeprom_buff);

	/* Update the checksum */
	if (ret_val == 0)
		TCALL(hw, eeprom.ops.update_checksum);

err:
	kfree(eeprom_buff);
	return ret_val;
}

static void txgbe_get_drvinfo(struct net_device *netdev,
			      struct ethtool_drvinfo *drvinfo)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);

	strscpy(drvinfo->driver, txgbe_driver_name,
		sizeof(drvinfo->driver));
	strscpy(drvinfo->version, txgbe_driver_version,
		sizeof(drvinfo->version));
	strscpy(drvinfo->fw_version, adapter->fw_version,
		sizeof(drvinfo->fw_version));
	strscpy(drvinfo->bus_info, pci_name(adapter->pdev),
		sizeof(drvinfo->bus_info));
	if (adapter->num_tx_queues <= TXGBE_NUM_RX_QUEUES) {
		drvinfo->n_stats = TXGBE_STATS_LEN -
				   (TXGBE_NUM_RX_QUEUES - adapter->num_tx_queues)*
					(sizeof(struct txgbe_queue_stats) / sizeof(u64))*2;
	}else{
		drvinfo->n_stats = TXGBE_STATS_LEN;
	}
	drvinfo->testinfo_len = TXGBE_TEST_LEN;
	drvinfo->regdump_len = txgbe_get_regs_len(netdev);
}

static void txgbe_get_ringparam(struct net_device *netdev,
#ifdef HAVE_ETHTOOL_EXTENDED_RINGPARAMS
				struct ethtool_ringparam *ring,
				struct kernel_ethtool_ringparam *ringp,
				struct netlink_ext_ack *extack)
#else
				struct ethtool_ringparam *ring)
#endif
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);

	ring->rx_max_pending = TXGBE_MAX_RXD;
	ring->tx_max_pending = TXGBE_MAX_TXD;
	ring->rx_mini_max_pending = 0;
	ring->rx_jumbo_max_pending = 0;
	ring->rx_pending = adapter->rx_ring_count;
	ring->tx_pending = adapter->tx_ring_count;
	ring->rx_mini_pending = 0;
	ring->rx_jumbo_pending = 0;
}

static int txgbe_set_ringparam(struct net_device *netdev,
#ifdef HAVE_ETHTOOL_EXTENDED_RINGPARAMS
				struct ethtool_ringparam *ring,
				struct kernel_ethtool_ringparam *ringp,
				struct netlink_ext_ack *extack)
#else
				struct ethtool_ringparam *ring)
#endif
{
	struct txgbe_ring *tx_ring = NULL, *rx_ring = NULL;
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 new_rx_count, new_tx_count;
	int i, j, err = 0;

	if ((ring->rx_mini_pending) || (ring->rx_jumbo_pending))
		return -EINVAL;

	if (hw->mac.type == txgbe_mac_aml || hw->mac.type == txgbe_mac_aml40) {
		new_tx_count = clamp_t(u32, ring->tx_pending,
					TXGBE_MIN_TXD_AML, TXGBE_MAX_TXD);
		new_tx_count = ALIGN(new_tx_count, TXGBE_REQ_TX_DESCRIPTOR_MULTIPLE);
	} else {
		new_tx_count = clamp_t(u32, ring->tx_pending,
					TXGBE_MIN_TXD, TXGBE_MAX_TXD);
		new_tx_count = ALIGN(new_tx_count, TXGBE_REQ_TX_DESCRIPTOR_MULTIPLE);
	}
	new_rx_count = clamp_t(u32, ring->rx_pending,
			       TXGBE_MIN_RXD, TXGBE_MAX_RXD);
	new_rx_count = ALIGN(new_rx_count, TXGBE_REQ_RX_DESCRIPTOR_MULTIPLE);

	if ((new_tx_count == adapter->tx_ring_count) &&
	    (new_rx_count == adapter->rx_ring_count)) {
		/* nothing to do */
		return 0;
	}

#ifdef HAVE_AF_XDP_ZC_SUPPORT
	/* If there is a AF_XDP UMEM attached to any of Rx rings,
	 * disallow changing the number of descriptors -- regardless
	 * if the netdev is running or not.
	 */
	if (txgbe_xsk_any_rx_ring_enabled(adapter))
		return -EBUSY;
#endif /* HAVE_AF_XDP_ZC_SUPPORT */

	while (test_and_set_bit(__TXGBE_RESETTING, &adapter->state))
		usleep_range(1000, 2000);

	if (!netif_running(adapter->netdev)) {
		for (i = 0; i < adapter->num_tx_queues; i++)
			adapter->tx_ring[i]->count = new_tx_count;
		for (i = 0; i < adapter->num_xdp_queues; i++)
			adapter->xdp_ring[i]->count = new_tx_count;
		for (i = 0; i < adapter->num_rx_queues; i++)
			adapter->rx_ring[i]->count = new_rx_count;
		adapter->tx_ring_count = new_tx_count;
		adapter->xdp_ring_count = new_tx_count;
		adapter->rx_ring_count = new_rx_count;
		goto done;
	}

	i = max_t(int, adapter->num_tx_queues + adapter->num_xdp_queues,
		  adapter->num_rx_queues);

	/*
	 * Setup new Tx resources and free the old Tx resources in that order.
	 * We can then assign the new resources to the rings via a memcpy.
	 * The advantage to this approach is that we are guaranteed to still
	 * have resources even in the case of an allocation failure.
	 */
	if (new_tx_count != adapter->tx_ring_count) {
		netdev_info(netdev,
			"Changing Tx descriptor count from %d to %d.\n",
			adapter->tx_ring[0]->count, new_tx_count);
		tx_ring = kcalloc(i, sizeof(struct txgbe_ring), GFP_KERNEL);
		if (!tx_ring) {
			err = -ENOMEM;
			goto done;
		}

		for (i = 0; i < adapter->num_tx_queues; i++) {
			memcpy(&tx_ring[i], adapter->tx_ring[i],
			       sizeof(struct txgbe_ring));

			tx_ring[i].count = new_tx_count;
			/* the desc and bi pointers will be reallocated
			 * in the setup call
			 */
			tx_ring[i].desc = NULL;
			tx_ring[i].tx_buffer_info = NULL;
			err = txgbe_setup_tx_resources(&tx_ring[i]);
			if (err) {
				while (i) {
					i--;
					txgbe_free_tx_resources(&tx_ring[i]);
				}

				kfree(tx_ring);
				tx_ring = NULL;
				err = -ENOMEM;

				goto done;
			}
		}

		for (j = 0; j < adapter->num_xdp_queues; j++, i++) {
			memcpy(&tx_ring[i], adapter->xdp_ring[j],
			       sizeof(struct txgbe_ring));

			tx_ring[i].count = new_tx_count;
			/* the desc and bi pointers will be reallocated
			 * in the setup call
			 */
			tx_ring[i].desc = NULL;
			tx_ring[i].tx_buffer_info = NULL;
			err = txgbe_setup_tx_resources(&tx_ring[i]);
			if (err) {
				while (i) {
					i--;
					txgbe_free_tx_resources(&tx_ring[i]);
				}

				kfree(tx_ring);
				tx_ring = NULL;
				err = -ENOMEM;

				goto done;
			}
		}
	}

	/* Repeat the process for the Rx rings if needed */
	if (new_rx_count != adapter->rx_ring_count) {
		netdev_info(netdev,
			"Changing Rx descriptor count from %d to %d\n",
			adapter->rx_ring[0]->count, new_rx_count);
		rx_ring = kcalloc(i, sizeof(struct txgbe_ring), GFP_KERNEL);
		if (!rx_ring) {
			err = -ENOMEM;
			goto free_tx;
		}

		for (i = 0; i < adapter->num_rx_queues; i++) {
			u16 unused;

			memcpy(&rx_ring[i], adapter->rx_ring[i],
			       sizeof(struct txgbe_ring));
#ifdef HAVE_XDP_BUFF_RXQ
			xdp_rxq_info_unreg(&rx_ring[i].xdp_rxq);
#endif
			rx_ring[i].count = new_rx_count;
			/* the desc and bi pointers will be reallocated
			 * in the setup call
			 */
			rx_ring[i].desc = NULL;
			rx_ring[i].rx_buffer_info = NULL;
			err = txgbe_setup_rx_resources(&rx_ring[i]);
			if (err)
				goto rx_unwind;

			unused = txgbe_desc_unused(&rx_ring[i]);
			err = txgbe_alloc_rx_buffers(&rx_ring[i], unused);
rx_unwind:
			if (err) {
				err = -ENOMEM;

				do {
					txgbe_free_rx_resources(&rx_ring[i]);
				} while (i--);
				kfree(rx_ring);
				rx_ring = NULL;

				goto free_tx;
			}
		}
	}

	/* Bring interface down, copy in the new ring info,
	 * then restore the interface
	 */
	txgbe_down(adapter);

	if (tx_ring) {
		for (i = 0; i < adapter->num_tx_queues; i++) {
			txgbe_free_tx_resources(adapter->tx_ring[i]);
			memcpy(adapter->tx_ring[i], &tx_ring[i],
				sizeof(struct txgbe_ring));
		}

		for (j = 0; j < adapter->num_xdp_queues; j++, i++) {
			txgbe_free_tx_resources(adapter->xdp_ring[j]);
			memcpy(adapter->xdp_ring[j], &tx_ring[i],
				sizeof(struct txgbe_ring));
		}

		kfree(tx_ring);
		tx_ring = NULL;
	}

	if (rx_ring) {
		for (i = 0; i < adapter->num_rx_queues; i++) {
			txgbe_free_rx_resources(adapter->rx_ring[i]);
			/* this is to fake out the allocation routine
			 * into thinking it has to realloc everything
			 * but the recycling logic will let us re-use
			 * the buffers allocated above
			 */
			rx_ring[i].next_to_use = 0;
			rx_ring[i].next_to_clean = 0;
			rx_ring[i].next_to_alloc = 0;
			/* do a struct copy */
			memcpy(adapter->rx_ring[i], &rx_ring[i],
				sizeof(struct txgbe_ring));
		}
		kfree(rx_ring);
		rx_ring = NULL;
	}

	adapter->tx_ring_count = new_tx_count;
	adapter->xdp_ring_count = new_tx_count;
	adapter->rx_ring_count = new_rx_count;

	txgbe_up(adapter);

free_tx:
/* error cleanup if the Rx allocations failed after getting Tx */
	if (tx_ring) {
		for (i = 0; i < adapter->num_tx_queues; i++) {
			txgbe_free_tx_resources(adapter->tx_ring[i]);
			memcpy(adapter->tx_ring[i], &tx_ring[i],
				sizeof(struct txgbe_ring));
		}

		for (j = 0; j < adapter->num_xdp_queues; j++, i++) {
			txgbe_free_tx_resources(adapter->xdp_ring[j]);
			memcpy(adapter->xdp_ring[j], &tx_ring[i],
				sizeof(struct txgbe_ring));
		}

		kfree(tx_ring);
		tx_ring = NULL;
	}

done:
	clear_bit(__TXGBE_RESETTING, &adapter->state);

	return err;
}

#ifndef HAVE_ETHTOOL_GET_SSET_COUNT
static int txgbe_get_stats_count(struct net_device *netdev)
{
	if (adapter->num_tx_queues <= TXGBE_NUM_RX_QUEUES) {
		return  TXGBE_STATS_LEN - (TXGBE_NUM_RX_QUEUES - adapter->num_tx_queues)*
					(sizeof(struct txgbe_queue_stats) / sizeof(u64))*2;
	}else{
		return TXGBE_STATS_LEN;
	}

}

#else /* HAVE_ETHTOOL_GET_SSET_COUNT */
static int txgbe_get_sset_count(struct net_device *netdev, int sset)
{
#ifdef HAVE_TX_MQ
#ifndef HAVE_NETDEV_SELECT_QUEUE
	struct txgbe_adapter *adapter = netdev_priv(netdev);
#endif
#endif
	struct txgbe_adapter *adapter = netdev_priv(netdev);

	switch (sset) {
	case ETH_SS_TEST:
		return TXGBE_TEST_LEN;
	case ETH_SS_STATS:
		if (adapter->num_tx_queues <= TXGBE_NUM_RX_QUEUES) {
			return  TXGBE_STATS_LEN - (TXGBE_NUM_RX_QUEUES - adapter->num_tx_queues)*
					(sizeof(struct txgbe_queue_stats) / sizeof(u64))*2;
		}else{
			return TXGBE_STATS_LEN;
		}
	case ETH_SS_PRIV_FLAGS:
		return TXGBE_PRIV_FLAGS_STR_LEN;
	default:
		return -EOPNOTSUPP;
	}
}

#ifdef HAVE_ETHTOOL_GET_SSET_COUNT
/**
 * txgbe_get_priv_flags - report device private flags
 * @dev: network interface device structure
 *
 * The get string set count and the string set should be matched for each
 * flag returned.  Add new strings for each flag to the txgbe_gstrings_priv_flags
 * array.
 *
 * Returns a u32 bitmap of flags.
 **/
static u32 txgbe_get_priv_flags(struct net_device *dev)
{
	struct txgbe_adapter *adapter = netdev_priv(dev);
	u32 i, ret_flags = 0;

	for (i = 0; i < TXGBE_PRIV_FLAGS_STR_LEN; i++) {
		const struct txgbe_priv_flags *priv_flags;

		priv_flags = &txgbe_gstrings_priv_flags[i];

		if (priv_flags->flag & adapter->eth_priv_flags)
			ret_flags |= BIT(i);
	}
	return ret_flags;
}

/**
 * txgbe_set_priv_flags - set private flags
 * @dev: network interface device structure
 * @flags: bit flags to be set
 **/
static int txgbe_set_priv_flags(struct net_device *dev, u32 flags)
{
	struct txgbe_adapter *adapter = netdev_priv(dev);
	u32 orig_flags, new_flags, changed_flags;
	bool reset_needed = 0;
	u32 i;
	s32 status = 0;

	orig_flags = adapter->eth_priv_flags;
	new_flags = orig_flags;

	for (i = 0; i < TXGBE_PRIV_FLAGS_STR_LEN; i++) {
		const struct txgbe_priv_flags *priv_flags;

		priv_flags = &txgbe_gstrings_priv_flags[i];

		if (flags & BIT(i))
			new_flags |= priv_flags->flag;
		else
			new_flags &= ~(priv_flags->flag);

		/* If this is a read-only flag, it can't be changed */
		if (priv_flags->read_only &&
		    ((orig_flags ^ new_flags) & BIT(i)))
			return -EOPNOTSUPP;
	}
	
	changed_flags = orig_flags ^ new_flags;

	if(!changed_flags) return 0;

	if (changed_flags & TXGBE_ETH_PRIV_FLAG_LLDP)
		reset_needed = 1;

	if (changed_flags & TXGBE_ETH_PRIV_FLAG_LLDP) {
		status = txgbe_hic_write_lldp(&adapter->hw, (u32)(new_flags & TXGBE_ETH_PRIV_FLAG_LLDP));
		if (!status)
			adapter->eth_priv_flags = new_flags;
	}

#ifdef HAVE_SWIOTLB_SKIP_CPU_SYNC
	if (changed_flags & TXGBE_ETH_PRIV_FLAG_LEGACY_RX) {
		adapter->eth_priv_flags = new_flags;

		if (adapter->eth_priv_flags & TXGBE_ETH_PRIV_FLAG_LEGACY_RX)
			adapter->flags2 |= TXGBE_FLAG2_RX_LEGACY;
		else
			adapter->flags2 &= ~TXGBE_FLAG2_RX_LEGACY;

		/* reset interface to repopulate queues */
		if (netif_running(dev))
			txgbe_reinit_locked(adapter);
	}
#endif

	return status;
}

#endif /*HAVE_ETHTOOL_GET_SSET_COUNT*/

#endif /* HAVE_ETHTOOL_GET_SSET_COUNT */
static void txgbe_get_ethtool_stats(struct net_device *netdev,
				    struct ethtool_stats __always_unused *stats,
				    u64 *data)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
#ifdef HAVE_NETDEV_STATS_IN_NETDEV
	struct net_device_stats *net_stats = &netdev->stats;
#else
	struct net_device_stats *net_stats = &adapter->net_stats;
#endif
#ifdef HAVE_NDO_GET_STATS64
	unsigned int start;
#endif
	struct txgbe_ring *ring;
	int i, j;
	char *p;

	txgbe_update_stats(adapter);

	for (i = 0; i < TXGBE_NETDEV_STATS_LEN; i++) {
		p = (char *)net_stats + txgbe_gstrings_net_stats[i].stat_offset;
		data[i] = (txgbe_gstrings_net_stats[i].sizeof_stat ==
			sizeof(u64)) ? *(u64 *)p : *(u32 *)p;
	}
	for (j = 0; j < TXGBE_GLOBAL_STATS_LEN; j++, i++) {
		p = (char *)adapter + txgbe_gstrings_stats[j].stat_offset;
		data[i] = (txgbe_gstrings_stats[j].sizeof_stat ==
			   sizeof(u64)) ? *(u64 *)p : *(u32 *)p;
	}
	for (j = 0; j < adapter->num_tx_queues; j++) {
		ring = adapter->tx_ring[j];
		if (!ring) {
			data[i++] = 0;
			data[i++] = 0;
#ifdef BP_EXTENDED_STATS
			data[i++] = 0;
			data[i++] = 0;
			data[i++] = 0;
#endif
			continue;
		}

#ifdef HAVE_NDO_GET_STATS64
		do {
			start = u64_stats_fetch_begin(&ring->syncp);
#endif
			data[i]   = ring->stats.packets;
			data[i+1] = ring->stats.bytes;
#ifdef HAVE_NDO_GET_STATS64
		} while (u64_stats_fetch_retry(&ring->syncp, start));
#endif
		i += 2;
#ifdef BP_EXTENDED_STATS
		data[i] = ring->stats.yields;
		data[i+1] = ring->stats.misses;
		data[i+2] = ring->stats.cleaned;
		i += 3;
#endif
	}
	for (j = 0; j < adapter->num_rx_queues; j++) {
		ring = adapter->rx_ring[j];
		if (!ring) {
			data[i++] = 0;
			data[i++] = 0;
#ifdef BP_EXTENDED_STATS
			data[i++] = 0;
			data[i++] = 0;
			data[i++] = 0;
#endif
			continue;
		}

#ifdef HAVE_NDO_GET_STATS64
		do {
			start = u64_stats_fetch_begin(&ring->syncp);
#endif
			data[i]   = ring->stats.packets;
			data[i+1] = ring->stats.bytes;
#ifdef HAVE_NDO_GET_STATS64
		} while (u64_stats_fetch_retry(&ring->syncp, start));
#endif
		i += 2;
#ifdef BP_EXTENDED_STATS
		data[i] = ring->stats.yields;
		data[i+1] = ring->stats.misses;
		data[i+2] = ring->stats.cleaned;
		i += 3;
#endif
	}
	for (j = 0; j < TXGBE_MAX_PACKET_BUFFERS; j++) {
		data[i++] = adapter->stats.pxontxc[j];
		data[i++] = adapter->stats.pxofftxc[j];
	}
	for (j = 0; j < TXGBE_MAX_PACKET_BUFFERS; j++) {
		data[i++] = adapter->stats.pxonrxc[j];
		data[i++] = adapter->stats.pxoffrxc[j];
	}
}

static void txgbe_get_priv_flag_strings(struct net_device *netdev, u8 *data)
{
	char *p = (char *)data;
	unsigned int i;

	for (i = 0; i < TXGBE_PRIV_FLAGS_STR_LEN; i++) {
		snprintf(p, ETH_GSTRING_LEN, "%s",
			txgbe_gstrings_priv_flags[i].flag_string);
		p += ETH_GSTRING_LEN;
	}
}

static void txgbe_get_strings(struct net_device *netdev, u32 stringset,
			      u8 *data)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	char *p = (char *)data;
	int i;

	switch (stringset) {
	case ETH_SS_TEST:
		memcpy(data, *txgbe_gstrings_test,
		       TXGBE_TEST_LEN * ETH_GSTRING_LEN);
		break;
	case ETH_SS_STATS:
		for (i = 0; i < TXGBE_NETDEV_STATS_LEN; i++) {
			memcpy(p, txgbe_gstrings_net_stats[i].stat_string,
			       ETH_GSTRING_LEN);
			p += ETH_GSTRING_LEN;
		}
		for (i = 0; i < TXGBE_GLOBAL_STATS_LEN; i++) {
			memcpy(p, txgbe_gstrings_stats[i].stat_string,
			       ETH_GSTRING_LEN);
			p += ETH_GSTRING_LEN;
		}
		for (i = 0; i < adapter->num_tx_queues; i++) {
			sprintf(p, "tx_queue_%u_packets", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "tx_queue_%u_bytes", i);
			p += ETH_GSTRING_LEN;
#ifdef BP_EXTENDED_STATS
			sprintf(p, "tx_queue_%u_bp_napi_yield", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "tx_queue_%u_bp_misses", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "tx_queue_%u_bp_cleaned", i);
			p += ETH_GSTRING_LEN;
#endif /* BP_EXTENDED_STATS */
		}
		for (i = 0; i < adapter->num_rx_queues; i++) {
			sprintf(p, "rx_queue_%u_packets", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "rx_queue_%u_bytes", i);
			p += ETH_GSTRING_LEN;
#ifdef BP_EXTENDED_STATS
			sprintf(p, "rx_queue_%u_bp_poll_yield", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "rx_queue_%u_bp_misses", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "rx_queue_%u_bp_cleaned", i);
			p += ETH_GSTRING_LEN;
#endif /* BP_EXTENDED_STATS */
		}
		for (i = 0; i < TXGBE_MAX_PACKET_BUFFERS; i++) {
			sprintf(p, "tx_pb_%u_pxon", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "tx_pb_%u_pxoff", i);
			p += ETH_GSTRING_LEN;
		}
		for (i = 0; i < TXGBE_MAX_PACKET_BUFFERS; i++) {
			sprintf(p, "rx_pb_%u_pxon", i);
			p += ETH_GSTRING_LEN;
			sprintf(p, "rx_pb_%u_pxoff", i);
			p += ETH_GSTRING_LEN;
		}
		/* BUG_ON(p - data != TXGBE_STATS_LEN * ETH_GSTRING_LEN); */
		break;
	case ETH_SS_PRIV_FLAGS:
		txgbe_get_priv_flag_strings(netdev, data);
		break;
	}
}

static int txgbe_link_test(struct txgbe_adapter *adapter, u64 *data)
{
	struct txgbe_hw *hw = &adapter->hw;
	bool link_up = false;
	u32 link_speed = 0;

	if (TXGBE_REMOVED(hw->hw_addr)) {
		*data = 1;
		return 1;
	}
	*data = 0;
    TCALL(hw, mac.ops.check_link, &link_speed, &link_up, true);
	if (link_up)
		return *data;
	else
		*data = 1;
	return *data;
}

/* ethtool register test data */
struct txgbe_reg_test {
	u32 reg;
	u8  array_len;
	u8  test_type;
	u32 mask;
	u32 write;
};

/* In the hardware, registers are laid out either singly, in arrays
 * spaced 0x40 bytes apart, or in contiguous tables.  We assume
 * most tests take place on arrays or single registers (handled
 * as a single-element array) and special-case the tables.
 * Table tests are always pattern tests.
 *
 * We also make provision for some required setup steps by specifying
 * registers to be written without any read-back testing.
 */

#define PATTERN_TEST    1
#define SET_READ_TEST   2
#define WRITE_NO_TEST   3
#define TABLE32_TEST    4
#define TABLE64_TEST_LO 5
#define TABLE64_TEST_HI 6

/* default sapphire register test */
static struct txgbe_reg_test reg_test_sapphire[] = {
	{ TXGBE_RDB_RFCL(0), 1, PATTERN_TEST, 0x8007FFF0, 0x8007FFF0 },
	{ TXGBE_RDB_RFCH(0), 1, PATTERN_TEST, 0x8007FFF0, 0x8007FFF0 },
	{ TXGBE_PSR_VLAN_CTL, 1, PATTERN_TEST, 0x00000000, 0x00000000 },
	{ TXGBE_PX_RR_BAL(0), 4, PATTERN_TEST, 0xFFFFFF80, 0xFFFFFF80 },
	{ TXGBE_PX_RR_BAH(0), 4, PATTERN_TEST, 0xFFFFFFFF, 0xFFFFFFFF },
	{ TXGBE_PX_RR_CFG(0), 4, WRITE_NO_TEST, 0, TXGBE_PX_RR_CFG_RR_EN },
	{ TXGBE_RDB_RFCH(0), 1, PATTERN_TEST, 0x8007FFF0, 0x8007FFF0 },
	{ TXGBE_RDB_RFCV(0), 1, PATTERN_TEST, 0xFFFFFFFF, 0xFFFFFFFF },
	{ TXGBE_PX_TR_BAL(0), 4, PATTERN_TEST, 0xFFFFFFFF, 0xFFFFFFFF },
	{ TXGBE_PX_TR_BAH(0), 4, PATTERN_TEST, 0xFFFFFFFF, 0xFFFFFFFF },
	{ TXGBE_RDB_PB_CTL, 1, SET_READ_TEST, 0x00000001, 0x00000001 },
	{ TXGBE_PSR_MC_TBL(0), 128, TABLE32_TEST, 0xFFFFFFFF, 0xFFFFFFFF },
	{ .reg = 0 }
};


static bool reg_pattern_test(struct txgbe_adapter *adapter, u64 *data, int reg,
			     u32 mask, u32 write)
{
	u32 pat, val, before;
	static const u32 test_pattern[] = {
		0x5A5A5A5A, 0xA5A5A5A5, 0x00000000, 0xFFFFFFFF
	};

	if (TXGBE_REMOVED(adapter->hw.hw_addr)) {
		*data = 1;
		return true;
	}
	for (pat = 0; pat < ARRAY_SIZE(test_pattern); pat++) {
		before = rd32(&adapter->hw, reg);
		wr32(&adapter->hw, reg, test_pattern[pat] & write);
		val = rd32(&adapter->hw, reg);
		if (val != (test_pattern[pat] & write & mask)) {
			e_err(drv,
			      "pattern test reg %04X failed: got 0x%08X "
			      "expected 0x%08X\n",
			      reg, val, test_pattern[pat] & write & mask);
			*data = reg;
			wr32(&adapter->hw, reg, before);
			return true;
		}
		wr32(&adapter->hw, reg, before);
	}
	return false;
}

static bool reg_set_and_check(struct txgbe_adapter *adapter, u64 *data, int reg,
			      u32 mask, u32 write)
{
	u32 val, before;

	if (TXGBE_REMOVED(adapter->hw.hw_addr)) {
		*data = 1;
		return true;
	}
	before = rd32(&adapter->hw, reg);
	wr32(&adapter->hw, reg, write & mask);
	val = rd32(&adapter->hw, reg);
	if ((write & mask) != (val & mask)) {
		e_err(drv,
		      "set/check reg %04X test failed: got 0x%08X expected"
		      "0x%08X\n",
		      reg, (val & mask), (write & mask));
		*data = reg;
		wr32(&adapter->hw, reg, before);
		return true;
	}
	wr32(&adapter->hw, reg, before);
	return false;
}

static bool txgbe_reg_test(struct txgbe_adapter *adapter, u64 *data)
{
	struct txgbe_reg_test *test;
	struct txgbe_hw *hw = &adapter->hw;
	u32 i;

	if (TXGBE_REMOVED(hw->hw_addr)) {
		e_err(drv, "Adapter removed - register test blocked\n");
		*data = 1;
		return true;
	}

	test = reg_test_sapphire;

	/*
	 * Perform the remainder of the register test, looping through
	 * the test table until we either fail or reach the null entry.
	 */
	while (test->reg) {
		for (i = 0; i < test->array_len; i++) {
			bool b = false;

			switch (test->test_type) {
			case PATTERN_TEST:
				b = reg_pattern_test(adapter, data,
						      test->reg + (i * 0x40),
						      test->mask,
						      test->write);
				break;
			case SET_READ_TEST:
				b = reg_set_and_check(adapter, data,
						       test->reg + (i * 0x40),
						       test->mask,
						       test->write);
				break;
			case WRITE_NO_TEST:
				wr32(hw, test->reg + (i * 0x40),
						test->write);
				break;
			case TABLE32_TEST:
				b = reg_pattern_test(adapter, data,
						      test->reg + (i * 4),
						      test->mask,
						      test->write);
				break;
			case TABLE64_TEST_LO:
				b = reg_pattern_test(adapter, data,
						      test->reg + (i * 8),
						      test->mask,
						      test->write);
				break;
			case TABLE64_TEST_HI:
				b = reg_pattern_test(adapter, data,
						      (test->reg + 4) + (i * 8),
						      test->mask,
						      test->write);
				break;
			}
			if (b)
				return true;
		}
		test++;
	}

	*data = 0;
	return false;
}

static bool txgbe_eeprom_test(struct txgbe_adapter *adapter, u64 *data)
{
	struct txgbe_hw *hw = &adapter->hw;

	if (TCALL(hw, eeprom.ops.validate_checksum, NULL)) {
		*data = 1;
		return true;
	} else {
		*data = 0;
		return false;
	}
}

static irqreturn_t txgbe_test_intr(int __always_unused irq, void *data)
{
	struct net_device *netdev = (struct net_device *) data;
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	u64 icr;

	/* get misc interrupt, as cannot get ring interrupt status */
	icr = txgbe_misc_isb(adapter, TXGBE_ISB_VEC1);
	icr <<= 32;
	icr |= txgbe_misc_isb(adapter, TXGBE_ISB_VEC0);

	adapter->test_icr = icr;

	return IRQ_HANDLED;
}

static int txgbe_intr_test(struct txgbe_adapter *adapter, u64 *data)
{
	struct net_device *netdev = adapter->netdev;
	u64 mask;
	u32 i = 0, shared_int = true;
	u32 irq = adapter->pdev->irq;

	if (TXGBE_REMOVED(adapter->hw.hw_addr)) {
		*data = 1;
		return -1;
	}
	*data = 0;

	txgbe_setup_isb_resources(adapter);
	txgbe_configure_isb(adapter);
	/* Hook up test interrupt handler just for this test */
	if (adapter->msix_entries) {
		/* NOTE: we don't test MSI-X interrupts here, yet */
		return 0;
	} else if (adapter->flags & TXGBE_FLAG_MSI_ENABLED) {
		shared_int = false;
		if (request_irq(irq, &txgbe_test_intr, 0, netdev->name,
				netdev)) {
			*data = 1;
			return -1;
		}
	} else if (!request_irq(irq, &txgbe_test_intr, IRQF_PROBE_SHARED,
				netdev->name, netdev)) {
		shared_int = false;
	} else if (request_irq(irq, &txgbe_test_intr, IRQF_SHARED,
			       netdev->name, netdev)) {
		*data = 1;
		return -1;
	}
	e_info(hw, "testing %s interrupt\n",
	       (shared_int ? "shared" : "unshared"));

	/* Disable all the interrupts */
	txgbe_irq_disable(adapter);
	TXGBE_WRITE_FLUSH(&adapter->hw);
	usleep_range(10000, 20000);

	/* Test each interrupt */
	for (; i < 1; i++) {
		/* Interrupt to test */
		mask = 1ULL << i;

		if (!shared_int) {
			/*
			 * Disable the interrupts to be reported in
			 * the cause register and then force the same
			 * interrupt and see if one gets posted.  If
			 * an interrupt was posted to the bus, the
			 * test failed.
			 */
			adapter->test_icr = 0;
			txgbe_intr_disable(&adapter->hw, ~mask);
			txgbe_intr_trigger(&adapter->hw, ~mask);
			TXGBE_WRITE_FLUSH(&adapter->hw);
			usleep_range(10000, 20000);

			if (adapter->test_icr & mask) {
				*data = 3;
				break;
			}
		}

		/*
		 * Enable the interrupt to be reported in the cause
		 * register and then force the same interrupt and see
		 * if one gets posted.  If an interrupt was not posted
		 * to the bus, the test failed.
		 */
		adapter->test_icr = 0;
		txgbe_intr_enable(&adapter->hw, mask);
		txgbe_intr_trigger(&adapter->hw, mask);
		TXGBE_WRITE_FLUSH(&adapter->hw);
		usleep_range(10000, 20000);

		if (!(adapter->test_icr & mask)) {
			*data = 4;
			break;
		}
	}

	/* Disable all the interrupts */
	txgbe_intr_disable(&adapter->hw, TXGBE_INTR_ALL);
	TXGBE_WRITE_FLUSH(&adapter->hw);
	usleep_range(10000, 20000);

	/* Unhook test interrupt handler */
	free_irq(irq, netdev);
	txgbe_free_isb_resources(adapter);
	
	return *data;
}

static void txgbe_free_desc_rings(struct txgbe_adapter *adapter)
{
	struct txgbe_ring *tx_ring = &adapter->test_tx_ring;
	struct txgbe_ring *rx_ring = &adapter->test_rx_ring;
	struct txgbe_hw *hw = &adapter->hw;

	/* shut down the DMA engines now so they can be reinitialized later */

	/* first Rx */
	TCALL(hw, mac.ops.disable_rx);
	txgbe_disable_rx_queue(adapter, rx_ring);

	/* now Tx */
	wr32(hw, TXGBE_PX_TR_CFG(tx_ring->reg_idx), 0);

	wr32m(hw, TXGBE_TDM_CTL, TXGBE_TDM_CTL_TE, 0);

	txgbe_reset(adapter);

	txgbe_free_tx_resources(&adapter->test_tx_ring);
	txgbe_free_rx_resources(&adapter->test_rx_ring);
}

static void txgbe_loopback_configure_tx_ring(struct txgbe_adapter *adapter,
			     struct txgbe_ring *ring)
{
	struct txgbe_hw *hw = &adapter->hw;
	u64 tdba = ring->dma;
	int wait_loop = 10;
	u32 txdctl = TXGBE_PX_TR_CFG_ENABLE;
	u8 reg_idx = ring->reg_idx;
#ifdef HAVE_AF_XDP_ZC_SUPPORT
	ring->xsk_pool = NULL;
	if (ring_is_xdp(ring))
		ring->xsk_pool = txgbe_xsk_umem(adapter, ring);
#endif
	/* disable queue to avoid issues while updating state */
	wr32(hw, TXGBE_PX_TR_CFG(reg_idx), TXGBE_PX_TR_CFG_SWFLSH);
	TXGBE_WRITE_FLUSH(hw);

	wr32(hw, TXGBE_PX_TR_BAL(reg_idx), tdba & DMA_BIT_MASK(32));
	wr32(hw, TXGBE_PX_TR_BAH(reg_idx), tdba >> 32);

	/* reset head and tail pointers */
	wr32(hw, TXGBE_PX_TR_RP(reg_idx), 0);
	wr32(hw, TXGBE_PX_TR_WP(reg_idx), 0);
	ring->tail = adapter->io_addr + TXGBE_PX_TR_WP(reg_idx);

	/* reset ntu and ntc to place SW in sync with hardwdare */
	ring->next_to_clean = 0;
	ring->next_to_use = 0;

	txdctl |= TXGBE_RING_SIZE(ring) << TXGBE_PX_TR_CFG_TR_SIZE_SHIFT;

	/*
	 * set WTHRESH to encourage burst writeback, it should not be set
	 * higher than 1 when:
	 * - ITR is 0 as it could cause false TX hangs
	 * - ITR is set to > 100k int/sec and BQL is enabled
	 *
	 * In order to avoid issues WTHRESH + PTHRESH should always be equal
	 * to or less than the number of on chip descriptors, which is
	 * currently 40.
	 */

	txdctl |= 0x20 << TXGBE_PX_TR_CFG_WTHRESH_SHIFT;

	/* reinitialize flowdirector state */
	if (adapter->flags & TXGBE_FLAG_FDIR_HASH_CAPABLE) {
		ring->atr_sample_rate = adapter->atr_sample_rate;
		ring->atr_count = 0;
		set_bit(__TXGBE_TX_FDIR_INIT_DONE, &ring->state);
	} else {
		ring->atr_sample_rate = 0;
	}

	/* initialize XPS */
	if (!test_and_set_bit(__TXGBE_TX_XPS_INIT_DONE, &ring->state)) {
		struct txgbe_q_vector *q_vector = ring->q_vector;

		if (q_vector)
			netif_set_xps_queue(adapter->netdev,
					    &q_vector->affinity_mask,
					    ring->queue_index);
	}

	clear_bit(__TXGBE_HANG_CHECK_ARMED, &ring->state);

	/* enable queue */
	wr32(hw, TXGBE_PX_TR_CFG(reg_idx), txdctl);

	/* poll to verify queue is enabled */
	do {
		msleep(20);
		txdctl = rd32(hw, TXGBE_PX_TR_CFG(reg_idx));
	} while (--wait_loop && !(txdctl & TXGBE_PX_TR_CFG_ENABLE));
	if (!wait_loop)
		e_err(drv, "Could not enable Tx Queue %d\n", reg_idx);
}


static int txgbe_setup_desc_rings(struct txgbe_adapter *adapter)
{
	struct txgbe_ring *tx_ring = &adapter->test_tx_ring;
	struct txgbe_ring *rx_ring = &adapter->test_rx_ring;
	struct txgbe_hw *hw = &adapter->hw;
	int ret_val;
	int err;

	TCALL(hw, mac.ops.setup_rxpba, 0, 0, PBA_STRATEGY_EQUAL);

	/* Setup Tx descriptor ring and Tx buffers */
	tx_ring->count = TXGBE_DEFAULT_TXD;
	tx_ring->queue_index = 0;
	tx_ring->dev = pci_dev_to_dev(adapter->pdev);
	tx_ring->netdev = adapter->netdev;
	tx_ring->reg_idx = adapter->tx_ring[0]->reg_idx;

	err = txgbe_setup_tx_resources(tx_ring);
	if (err)
		return 1;

	wr32m(&adapter->hw, TXGBE_TDM_CTL,
		TXGBE_TDM_CTL_TE, TXGBE_TDM_CTL_TE);

	txgbe_loopback_configure_tx_ring(adapter, tx_ring);

	/* enable mac transmitter */

	if (hw->mac.type == txgbe_mac_aml40) {
		wr32(hw, TXGBE_MAC_TX_CFG, (rd32(hw, TXGBE_MAC_TX_CFG) &
				~TXGBE_MAC_TX_CFG_AML_SPEED_MASK) | TXGBE_MAC_TX_CFG_TE |
				TXGBE_MAC_TX_CFG_AML_SPEED_40G);
	} else if (hw->mac.type == txgbe_mac_aml) {
		if ((rd32(hw, TXGBE_CFG_PORT_ST) & TXGBE_CFG_PORT_ST_AML_LINK_10G) ==
						TXGBE_CFG_PORT_ST_AML_LINK_10G)
			wr32(hw, TXGBE_MAC_TX_CFG, (rd32(hw, TXGBE_MAC_TX_CFG) &
					~TXGBE_MAC_TX_CFG_AML_SPEED_MASK) | TXGBE_MAC_TX_CFG_TE |
					TXGBE_MAC_TX_CFG_AML_SPEED_10G);
		else
			wr32(hw, TXGBE_MAC_TX_CFG, (rd32(hw, TXGBE_MAC_TX_CFG) &
					~TXGBE_MAC_TX_CFG_AML_SPEED_MASK) | TXGBE_MAC_TX_CFG_TE |
					TXGBE_MAC_TX_CFG_AML_SPEED_25G);
	} else {
		if(txgbe_check_reset_blocked(hw) && (hw->phy.autoneg_advertised == TXGBE_LINK_SPEED_1GB_FULL ||
						     adapter->link_speed == TXGBE_LINK_SPEED_1GB_FULL))
			wr32m(hw, TXGBE_MAC_TX_CFG,
				TXGBE_MAC_TX_CFG_TE | TXGBE_MAC_TX_CFG_SPEED_MASK,
				TXGBE_MAC_TX_CFG_TE | TXGBE_MAC_TX_CFG_SPEED_1G);
		else
			wr32m(hw, TXGBE_MAC_TX_CFG,
				TXGBE_MAC_TX_CFG_TE | TXGBE_MAC_TX_CFG_SPEED_MASK,
				TXGBE_MAC_TX_CFG_TE | TXGBE_MAC_TX_CFG_SPEED_10G);
	}

	/* Setup Rx Descriptor ring and Rx buffers */
	rx_ring->count = TXGBE_DEFAULT_RXD;
	rx_ring->queue_index = 0;
	rx_ring->dev = pci_dev_to_dev(adapter->pdev);
	rx_ring->netdev = adapter->netdev;
	rx_ring->reg_idx = adapter->rx_ring[0]->reg_idx;
#ifdef CONFIG_TXGBE_DISABLE_PACKET_SPLIT
	rx_ring->rx_buf_len = TXGBE_RXBUFFER_2K;
#endif

	err = txgbe_setup_rx_resources(rx_ring);
	if (err) {
		ret_val = 4;
		goto err_nomem;
	}

	TCALL(hw, mac.ops.disable_rx);

	txgbe_configure_rx_ring(adapter, rx_ring);

	TCALL(hw, mac.ops.enable_rx);

	return 0;

err_nomem:
	txgbe_free_desc_rings(adapter);
	return ret_val;
}

static int txgbe_setup_config(struct txgbe_adapter *adapter)
{
	struct txgbe_hw *hw = &adapter->hw;
	u32 reg_data;

	/* Setup traffic loopback */
	reg_data = rd32(hw, TXGBE_PSR_CTL);
	reg_data |= TXGBE_PSR_CTL_BAM | TXGBE_PSR_CTL_UPE |
		TXGBE_PSR_CTL_MPE | TXGBE_PSR_CTL_TPE;
	wr32(hw, TXGBE_PSR_CTL, reg_data);

	wr32(hw, TXGBE_RSC_CTL,
		(rd32(hw, TXGBE_RSC_CTL) |
		TXGBE_RSC_CTL_SAVE_MAC_ERR) & ~TXGBE_RSC_CTL_SECRX_DIS);

	wr32(hw, TXGBE_RSC_LSEC_CTL, 0x4);

	wr32(hw, TXGBE_PSR_VLAN_CTL,
		rd32(hw, TXGBE_PSR_VLAN_CTL) &
		~TXGBE_PSR_VLAN_CTL_VFE);

	wr32m(&adapter->hw, TXGBE_MAC_RX_CFG,
		TXGBE_MAC_RX_CFG_LM, ~TXGBE_MAC_RX_CFG_LM);
	wr32m(&adapter->hw, TXGBE_CFG_PORT_CTL,
		TXGBE_CFG_PORT_CTL_FORCE_LKUP, ~TXGBE_CFG_PORT_CTL_FORCE_LKUP);

	/* enable mac transmitter */
	if (hw->mac.type == txgbe_mac_aml || hw->mac.type == txgbe_mac_aml40) {
		wr32m(hw, TXGBE_TSC_CTL,
			TXGBE_TSC_CTL_TX_DIS | TXGBE_TSC_MACTX_AFIFO_RD_WTRMRK, 0xd0000);

		wr32m(hw, TXGBE_RSC_CTL,
			TXGBE_RSC_CTL_RX_DIS, 0);
	}

	TXGBE_WRITE_FLUSH(hw);
	usleep_range(10000, 20000);

	return 0;
}

static int txgbe_setup_mac_loopback_test(struct txgbe_adapter *adapter)
{
	wr32m(&adapter->hw, TXGBE_MAC_RX_CFG,
		TXGBE_MAC_RX_CFG_LM | TXGBE_MAC_RX_CFG_RE,
		TXGBE_MAC_RX_CFG_LM | TXGBE_MAC_RX_CFG_RE);

	wr32m(&adapter->hw, TXGBE_CFG_PORT_CTL,
		TXGBE_CFG_PORT_CTL_FORCE_LKUP, TXGBE_CFG_PORT_CTL_FORCE_LKUP);

	return 0;
}

static void txgbe_mac_loopback_cleanup(struct txgbe_adapter *adapter)
{
	wr32m(&adapter->hw, TXGBE_TSC_CTL,
		TXGBE_TSC_MACTX_AFIFO_RD_WTRMRK, 0x20000);
	wr32m(&adapter->hw, TXGBE_MAC_RX_CFG,
		TXGBE_MAC_RX_CFG_LM, ~TXGBE_MAC_RX_CFG_LM);
	wr32m(&adapter->hw, TXGBE_CFG_PORT_CTL,
		TXGBE_CFG_PORT_CTL_FORCE_LKUP, ~TXGBE_CFG_PORT_CTL_FORCE_LKUP);
}

static int txgbe_setup_phy_loopback_test(struct txgbe_adapter *adapter)
{
	struct txgbe_hw *hw = &adapter->hw;
	u32 value;
	/* setup phy loopback */
	value = txgbe_rd32_epcs(hw, TXGBE_PHY_MISC_CTL0);
	value |= TXGBE_PHY_MISC_CTL0_TX2RX_LB_EN_0 |
		TXGBE_PHY_MISC_CTL0_TX2RX_LB_EN_3_1;

	txgbe_wr32_epcs(hw, TXGBE_PHY_MISC_CTL0, value);

	value = txgbe_rd32_epcs(hw, TXGBE_SR_PMA_MMD_CTL1);
	txgbe_wr32_epcs(hw, TXGBE_SR_PMA_MMD_CTL1,
		value | TXGBE_SR_PMA_MMD_CTL1_LB_EN);
	return 0;
}

static void txgbe_phy_loopback_cleanup(struct txgbe_adapter *adapter)
{
	struct txgbe_hw *hw = &adapter->hw;
	u32 value;

	value = txgbe_rd32_epcs(hw, TXGBE_PHY_MISC_CTL0);
	value &= ~(TXGBE_PHY_MISC_CTL0_TX2RX_LB_EN_0 |
		TXGBE_PHY_MISC_CTL0_TX2RX_LB_EN_3_1);

	txgbe_wr32_epcs(hw, TXGBE_PHY_MISC_CTL0, value);
	value = txgbe_rd32_epcs(hw, TXGBE_SR_PMA_MMD_CTL1);
	txgbe_wr32_epcs(hw, TXGBE_SR_PMA_MMD_CTL1,
		value & ~TXGBE_SR_PMA_MMD_CTL1_LB_EN);
}


static void txgbe_create_lbtest_frame(struct sk_buff *skb,
				      unsigned int frame_size)
{
	memset(skb->data, 0xFF, frame_size);
	frame_size >>= 1;
	memset(&skb->data[frame_size], 0xAA, frame_size / 2 - 1);
	memset(&skb->data[frame_size + 10], 0xBE, 1);
	memset(&skb->data[frame_size + 12], 0xAF, 1);
}

static bool txgbe_check_lbtest_frame(struct txgbe_rx_buffer *rx_buffer,
				     unsigned int frame_size)
{
	unsigned char *data;
	bool match = true;

	frame_size >>= 1;

#ifdef CONFIG_TXGBE_DISABLE_PACKET_SPLIT
	data = rx_buffer->skb->data;
#else
	data = kmap(rx_buffer->page) + rx_buffer->page_offset;
#endif

	if (data[3] != 0xFF ||
	    data[frame_size + 10] != 0xBE ||
	    data[frame_size + 12] != 0xAF)
		match = false;

#ifndef CONFIG_TXGBE_DISABLE_PACKET_SPLIT
	kunmap(rx_buffer->page);

#endif
	return match;
}

static u16 txgbe_clean_test_rings(struct txgbe_ring *rx_ring,
				  struct txgbe_ring *tx_ring,
				  unsigned int size)
{
	union txgbe_rx_desc *rx_desc;
	struct txgbe_rx_buffer *rx_buffer;
	struct txgbe_tx_buffer *tx_buffer;
#ifdef CONFIG_TXGBE_DISABLE_PACKET_SPLIT
	const int bufsz = rx_ring->rx_buf_len;
#else
	const int bufsz = txgbe_rx_bufsz(rx_ring);
#endif
	u16 rx_ntc, tx_ntc, count = 0;

	/* initialize next to clean and descriptor values */
	rx_ntc = rx_ring->next_to_clean;
	tx_ntc = tx_ring->next_to_clean;
	rx_desc = TXGBE_RX_DESC(rx_ring, rx_ntc);

	while (txgbe_test_staterr(rx_desc, TXGBE_RXD_STAT_DD)) {
		/* unmap buffer on Tx side */
		tx_buffer = &tx_ring->tx_buffer_info[tx_ntc];
		txgbe_unmap_and_free_tx_resource(tx_ring, tx_buffer);

		/* check Rx buffer */
		rx_buffer = &rx_ring->rx_buffer_info[rx_ntc];

		/* sync Rx buffer for CPU read */
		dma_sync_single_for_cpu(rx_ring->dev,
#ifndef CONFIG_TXGBE_DISABLE_PACKET_SPLIT
					rx_buffer->page_dma,
#else
					rx_buffer->dma,
#endif
					bufsz,
					DMA_FROM_DEVICE);

		/* verify contents of skb */
		if (txgbe_check_lbtest_frame(rx_buffer, size))
			count++;

		/* sync Rx buffer for device write */
		dma_sync_single_for_device(rx_ring->dev,
#ifndef CONFIG_TXGBE_DISABLE_PACKET_SPLIT
					rx_buffer->page_dma,
#else
					rx_buffer->dma,
#endif
					bufsz,
					DMA_FROM_DEVICE);

		/* increment Rx/Tx next to clean counters */
		rx_ntc++;
		if (rx_ntc == rx_ring->count)
			rx_ntc = 0;
		tx_ntc++;
		if (tx_ntc == tx_ring->count)
			tx_ntc = 0;

		/* fetch next descriptor */
		rx_desc = TXGBE_RX_DESC(rx_ring, rx_ntc);
	}

	/* re-map buffers to ring, store next to clean values */
	txgbe_alloc_rx_buffers(rx_ring, count);
	rx_ring->next_to_clean = rx_ntc;
	tx_ring->next_to_clean = tx_ntc;

	return count;
}

static int txgbe_run_loopback_test(struct txgbe_adapter *adapter)
{
	struct txgbe_ring *tx_ring = &adapter->test_tx_ring;
	struct txgbe_ring *rx_ring = &adapter->test_rx_ring;
	int i, j, lc, good_cnt, ret_val = 0;
	unsigned int size = 1024;
	netdev_tx_t tx_ret_val;
	struct sk_buff *skb;
	u32 flags_orig = adapter->flags;


	/* DCB can modify the frames on Tx */
	adapter->flags &= ~TXGBE_FLAG_DCB_ENABLED;

	/* allocate test skb */
	skb = alloc_skb(size, GFP_KERNEL);
	if (!skb)
		return 11;

	/* place data into test skb */
	txgbe_create_lbtest_frame(skb, size);
	skb_put(skb, size);

	/*
	 * Calculate the loop count based on the largest descriptor ring
	 * The idea is to wrap the largest ring a number of times using 64
	 * send/receive pairs during each loop
	 */

	if (rx_ring->count <= tx_ring->count)
		lc = ((tx_ring->count / 64) * 2) + 1;
	else
		lc = ((rx_ring->count / 64) * 2) + 1;

	for (j = 0; j <= lc; j++) {
		/* reset count of good packets */
		good_cnt = 0;

		/* place 64 packets on the transmit queue*/
		for (i = 0; i < 64; i++) {
			skb_get(skb);
			tx_ret_val = txgbe_xmit_frame_ring(skb,
							   adapter,
							   tx_ring);
			if (tx_ret_val == NETDEV_TX_OK)
				good_cnt++;
		}

		if (good_cnt != 64) {
			ret_val = 12;
			break;
		}

		/* allow 200 milliseconds for packets to go from Tx to Rx */
		msleep(200);

		good_cnt = txgbe_clean_test_rings(rx_ring, tx_ring, size);
		if(j == 0)
			continue;
		else if (good_cnt != 64) {
			ret_val = 13;
			break;
		}
	}

	/* free the original skb */
	kfree_skb(skb);
	adapter->flags = flags_orig;

	return ret_val;
}

static int txgbe_loopback_test(struct txgbe_adapter *adapter, u64 *data)
{
	struct txgbe_hw *hw = &adapter->hw;
	/* Let firmware know the driver has taken over */
	wr32m(&adapter->hw, TXGBE_CFG_PORT_CTL,
			TXGBE_CFG_PORT_CTL_DRV_LOAD, TXGBE_CFG_PORT_CTL_DRV_LOAD);
	*data = txgbe_setup_config(adapter);
	if (*data)
		goto err_loopback;

	if (hw->mac.type == txgbe_mac_aml || hw->mac.type == txgbe_mac_aml40)
		*data = txgbe_setup_mac_loopback_test(adapter);
	else
		*data = txgbe_setup_phy_loopback_test(adapter);
	if (*data)
			goto err_loopback;

	*data = txgbe_setup_desc_rings(adapter);
	if (*data)
		goto out;
	*data = txgbe_run_loopback_test(adapter);
	if (*data)
			e_info(hw, "phy loopback testing failed\n");
	if (hw->mac.type == txgbe_mac_aml || hw->mac.type == txgbe_mac_aml40)
		txgbe_mac_loopback_cleanup(adapter);
	else
		txgbe_phy_loopback_cleanup(adapter);

err_loopback:
	txgbe_free_desc_rings(adapter);
out:
	/* Let firmware take over control of h/w */
	wr32m(&adapter->hw, TXGBE_CFG_PORT_CTL,
		TXGBE_CFG_PORT_CTL_DRV_LOAD, 0);

	return *data;
}

#ifndef HAVE_ETHTOOL_GET_SSET_COUNT
static int txgbe_diag_test_count(struct net_device __always_unused *netdev)
{
	return TXGBE_TEST_LEN;
}

#endif /* HAVE_ETHTOOL_GET_SSET_COUNT */
static void txgbe_diag_test(struct net_device *netdev,
			    struct ethtool_test *eth_test, u64 *data)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	bool if_running = netif_running(netdev);
	struct txgbe_hw *hw = &adapter->hw;

	if (TXGBE_REMOVED(hw->hw_addr)) {
		e_err(hw, "Adapter removed - test blocked\n");
		data[0] = 1;
		data[1] = 1;
		data[2] = 1;
		data[3] = 1;
		data[4] = 1;
		eth_test->flags |= ETH_TEST_FL_FAILED;
		return;
	}
	set_bit(__TXGBE_TESTING, &adapter->state);
	if (eth_test->flags == ETH_TEST_FL_OFFLINE) {
		if (adapter->flags & TXGBE_FLAG_SRIOV_ENABLED) {
			int i;
			for (i = 0; i < adapter->num_vfs; i++) {
				if (adapter->vfinfo[i].clear_to_send) {
					e_warn(drv, "Please take active VFS "
					       "offline and restart the "
					       "adapter before running NIC "
					       "diagnostics\n");
					data[0] = 1;
					data[1] = 1;
					data[2] = 1;
					data[3] = 1;
					data[4] = 1;
					eth_test->flags |= ETH_TEST_FL_FAILED;
					clear_bit(__TXGBE_TESTING,
						  &adapter->state);
					goto skip_ol_tests;
				}
			}
		}

		/* Offline tests */
		e_info(hw, "offline testing starting\n");

		/* Link test performed before hardware reset so autoneg doesn't
		 * interfere with test result */
		if (txgbe_link_test(adapter, &data[4]))
			eth_test->flags |= ETH_TEST_FL_FAILED;

		if (if_running)
			/* indicate we're in test mode */
			txgbe_close(netdev);
		else
			txgbe_reset(adapter);

		e_info(hw, "register testing starting\n");
		if (txgbe_reg_test(adapter, &data[0]))
			eth_test->flags |= ETH_TEST_FL_FAILED;
		txgbe_reset(adapter);
		e_info(hw, "eeprom testing starting\n");
		if (txgbe_eeprom_test(adapter, &data[1]))
			eth_test->flags |= ETH_TEST_FL_FAILED;

		txgbe_reset(adapter);
		e_info(hw, "interrupt testing starting\n");
		if (txgbe_intr_test(adapter, &data[2]))
			eth_test->flags |= ETH_TEST_FL_FAILED;

		if (((hw->subsystem_device_id & TXGBE_NCSI_MASK) == TXGBE_NCSI_SUP) ||
		    ((hw->subsystem_device_id & TXGBE_WOL_MASK) == TXGBE_WOL_SUP)){
			e_info(hw, "skip MAC loopback diagnostic when veto set\n");
			data[3] = 0;
			goto skip_loopback;
		}
		/* If SRIOV or VMDq is enabled then skip MAC
		 * loopback diagnostic. */
		if (adapter->flags & (TXGBE_FLAG_SRIOV_ENABLED |
				      TXGBE_FLAG_VMDQ_ENABLED)) {
			e_info(hw, "skip MAC loopback diagnostic in VT mode\n");
			data[3] = 0;
			goto skip_loopback;
		}

		txgbe_reset(adapter);
		e_info(hw, "loopback testing starting\n");
		if (txgbe_loopback_test(adapter, &data[3]))
			eth_test->flags |= ETH_TEST_FL_FAILED;

skip_loopback:
		txgbe_reset(adapter);

		/* clear testing bit and return adapter to previous state */
		clear_bit(__TXGBE_TESTING, &adapter->state);
		if (if_running)
			txgbe_open(netdev);
	} else {
		e_info(hw, "online testing starting\n");

		/* Online tests */
		if (txgbe_link_test(adapter, &data[4]))
			eth_test->flags |= ETH_TEST_FL_FAILED;

		/* Offline tests aren't run; pass by default */
		data[0] = 0;
		data[1] = 0;
		data[2] = 0;
		data[3] = 0;

		clear_bit(__TXGBE_TESTING, &adapter->state);
	}

skip_ol_tests:
	msleep_interruptible(4 * 1000);
}

static void txgbe_get_wol(struct net_device *netdev,
			  struct ethtool_wolinfo *wol)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;

	wol->supported = 0;
	wol->wolopts = 0;

	if ((hw->subsystem_device_id & TXGBE_WOL_MASK) != TXGBE_WOL_SUP)
		return;

	wol->supported = WAKE_MAGIC;

	if (!device_can_wakeup(pci_dev_to_dev(adapter->pdev)))
		return;

	if (adapter->wol & TXGBE_PSR_WKUP_CTL_EX)
		wol->wolopts |= WAKE_UCAST;
	if (adapter->wol & TXGBE_PSR_WKUP_CTL_MC)
		wol->wolopts |= WAKE_MCAST;
	if (adapter->wol & TXGBE_PSR_WKUP_CTL_BC)
		wol->wolopts |= WAKE_BCAST;
	if (adapter->wol & TXGBE_PSR_WKUP_CTL_MAG)
		wol->wolopts |= WAKE_MAGIC;
}

static int txgbe_set_wol(struct net_device *netdev, struct ethtool_wolinfo *wol)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;

	if (wol->wolopts & (WAKE_PHY | WAKE_ARP | WAKE_MAGICSECURE))
		return -EOPNOTSUPP;

	if((hw->subsystem_device_id & TXGBE_WOL_MASK) != TXGBE_WOL_SUP)
		return -EOPNOTSUPP;

	adapter->wol = 0;

	if (wol->wolopts & WAKE_MAGIC)
		adapter->wol |= TXGBE_PSR_WKUP_CTL_MAG;

	hw->wol_enabled = !!(adapter->wol);
	wr32(hw, TXGBE_PSR_WKUP_CTL, adapter->wol);

	device_set_wakeup_enable(pci_dev_to_dev(adapter->pdev), adapter->wol);

	return 0;

	return -EOPNOTSUPP;
}

static int txgbe_nway_reset(struct net_device *netdev)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	if (netif_running(netdev))
		txgbe_reinit_locked(adapter);

	return 0;
}

#ifdef HAVE_ETHTOOL_SET_PHYS_ID
static int txgbe_set_phys_id(struct net_device *netdev,
			     enum ethtool_phys_id_state state)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u16 value = 0;
	
	switch (state) {
	case ETHTOOL_ID_ACTIVE:
		if (hw->mac.type == txgbe_mac_aml || (hw->mac.type == txgbe_mac_aml40))
			txgbe_hic_notify_led_active(hw, 1);
		adapter->led_reg = rd32(hw, TXGBE_CFG_LED_CTL);
		return 2;

	case ETHTOOL_ID_ON:
		if ((hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1bd4) ||
			(hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1ff9)) {
			if (adapter->link_up) {
				switch (adapter->link_speed) {
				case TXGBE_LINK_SPEED_10GB_FULL:
					TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_10G);
					break;
				case TXGBE_LINK_SPEED_1GB_FULL:
					TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_1G);
					break;
				case TXGBE_LINK_SPEED_100_FULL:
					TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_100M);
					break;
				default:
					break;
				}
			} else 
				TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_10G);
		} else
			TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_UP);
		break;

	case ETHTOOL_ID_OFF:
		if ((hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1bd4) ||
			(hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1ff9)) {
			if (adapter->link_up) {
				switch (adapter->link_speed) {
				case TXGBE_LINK_SPEED_10GB_FULL:
					TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_10G);
					break;
				case TXGBE_LINK_SPEED_1GB_FULL:
					TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_1G);
					break;
				case TXGBE_LINK_SPEED_100_FULL:
					TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_100M);
					break;
				default:
					break;
				}
			} else 
				TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_10G);
		} else
			TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_UP);
		break;

	case ETHTOOL_ID_INACTIVE:
		/* Restore LED settings */
		if (hw->mac.type == txgbe_mac_aml || (hw->mac.type == txgbe_mac_aml40))
			txgbe_hic_notify_led_active(hw, 0);
		wr32(&adapter->hw, TXGBE_CFG_LED_CTL,
				adapter->led_reg);
		if ((hw->subsystem_device_id & 0xF0) == TXGBE_ID_XAUI) {
			txgbe_read_mdio(&hw->phy_dev, hw->phy.addr, 31, 0xF021, &value);
			txgbe_write_mdio(&hw->phy_dev, hw->phy.addr, 31, 0xF021, (value & 0xFFFC) | 0x0);
		}
		break;
	}

	return 0;
}
#else
static int txgbe_phys_id(struct net_device *netdev, u32 data)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 led_reg = rd32(hw, TXGBE_CFG_LED_CTL);
	u32 i;

	if (!data || data > 300)
		data = 300;

	for (i = 0; i < (data * 1000); i += 400) {
		if ((hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1bd4) ||
			(hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1ff9)){
			if (adapter->link_up) {
				switch (adapter->link_speed) {
				case TXGBE_LINK_SPEED_10GB_FULL:
					TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_10G);
					break;
				case TXGBE_LINK_SPEED_1GB_FULL:
					TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_1G);
					break;
				case TXGBE_LINK_SPEED_100_FULL:
					TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_100M);
					break;
				default:
					break;
				}
			} else 
				TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_10G);
		} else
			TCALL(hw, mac.ops.led_on, TXGBE_LED_LINK_UP);
		msleep_interruptible(200);
		if ((hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1bd4) ||
			(hw->oem_ssid != 0x0085 && hw->oem_svid == 0x1ff9)) {
			if (adapter->link_up) {
				switch (adapter->link_speed) {
				case TXGBE_LINK_SPEED_10GB_FULL:
					TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_10G);
					break;
				case TXGBE_LINK_SPEED_1GB_FULL:
					TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_1G);
					break;
				case TXGBE_LINK_SPEED_100_FULL:
					TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_100M);
					break;
				default:
					break;
				}
			} else 
				TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_10G);
		} else
			TCALL(hw, mac.ops.led_off, TXGBE_LED_LINK_UP);
		msleep_interruptible(200);
	}

	/* Restore LED settings */
	wr32(hw, TXGBE_CFG_LED_CTL, led_reg);

	return 0;
}
#endif /* HAVE_ETHTOOL_SET_PHYS_ID */

static int txgbe_get_coalesce(struct net_device *netdev,
#ifdef HAVE_ETHTOOL_COALESCE_EXTACK
			      struct ethtool_coalesce *ec,
			      struct kernel_ethtool_coalesce *kernel_coal,
			      struct netlink_ext_ack *extack)
#else
			      struct ethtool_coalesce *ec)
#endif
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);

	ec->tx_max_coalesced_frames_irq = adapter->tx_work_limit;
	/* only valid if in constant ITR mode */
	if (adapter->rx_itr_setting <= 1)
		ec->rx_coalesce_usecs = adapter->rx_itr_setting;
	else
		ec->rx_coalesce_usecs = adapter->rx_itr_setting >> 2;

	if (adapter->rx_itr_setting == 1)
		ec->use_adaptive_rx_coalesce = 1;

	/* if in mixed tx/rx queues per vector mode, report only rx settings */
	if (adapter->q_vector[0]->tx.count && adapter->q_vector[0]->rx.count)
		return 0;

	/* only valid if in constant ITR mode */
	if (adapter->tx_itr_setting <= 1)
		ec->tx_coalesce_usecs = adapter->tx_itr_setting;
	else
		ec->tx_coalesce_usecs = adapter->tx_itr_setting >> 2;

	return 0;
}

/*
 * this function must be called before setting the new value of
 * rx_itr_setting
 */
static bool txgbe_update_rsc(struct txgbe_adapter *adapter)
{
	struct net_device *netdev = adapter->netdev;

	/* nothing to do if LRO or RSC are not enabled */
	if (!(adapter->flags2 & TXGBE_FLAG2_RSC_CAPABLE) ||
	    !(netdev->features & NETIF_F_LRO))
		return false;

	/* check the feature flag value and enable RSC if necessary */
	if (adapter->rx_itr_setting == 1 ||
	    adapter->rx_itr_setting > TXGBE_MIN_RSC_ITR) {
		if (!(adapter->flags2 & TXGBE_FLAG2_RSC_ENABLED)) {
			adapter->flags2 |= TXGBE_FLAG2_RSC_ENABLED;
			e_info(probe, "rx-usecs value high enough "
				      "to re-enable RSC\n");
			return true;
		}
	/* if interrupt rate is too high then disable RSC */
	} else if (adapter->flags2 & TXGBE_FLAG2_RSC_ENABLED) {
		adapter->flags2 &= ~TXGBE_FLAG2_RSC_ENABLED;
#ifdef TXGBE_NO_LRO
		e_info(probe, "rx-usecs set too low, disabling RSC\n");
#else
		e_info(probe, "rx-usecs set too low, "
			      "falling back to software LRO\n");
#endif
		return true;
	}
	return false;
}

static int txgbe_set_coalesce(struct net_device *netdev,
#ifdef HAVE_ETHTOOL_COALESCE_EXTACK
			      struct ethtool_coalesce *ec,
			      struct kernel_ethtool_coalesce *kernel_coal,
			      struct netlink_ext_ack *extack)
#else
			      struct ethtool_coalesce *ec)
#endif
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
	struct txgbe_q_vector *q_vector;
	int i;
	u16 tx_itr_param, rx_itr_param;
	u16  tx_itr_prev;
	bool need_reset = false;
#if 0
	if(ec->tx_max_coalesced_frames_irq == adapter->tx_work_limit &&
	   ((adapter->rx_itr_setting <= 1) ? (ec->rx_coalesce_usecs == adapter->rx_itr_setting) :
	    (ec->rx_coalesce_usecs == adapter->rx_itr_setting >> 2))) {
		e_info(probe, "no coalesce parameters changed, aborting\n");
		return -EINVAL;
	}
#endif
	if (adapter->q_vector[0]->tx.count && adapter->q_vector[0]->rx.count) {
		/* reject Tx specific changes in case of mixed RxTx vectors */
		if (ec->tx_coalesce_usecs)
			return -EINVAL;
		tx_itr_prev = adapter->rx_itr_setting;
	} else {
		tx_itr_prev = adapter->tx_itr_setting;
	}

	if (ec->tx_max_coalesced_frames_irq) {
		if (ec->tx_max_coalesced_frames_irq <= TXGBE_MAX_TX_WORK)
			adapter->tx_work_limit = ec->tx_max_coalesced_frames_irq;
		else
			return -EINVAL;
	} else
		return -EINVAL;

	if ((ec->rx_coalesce_usecs > (TXGBE_MAX_EITR >> 2)) ||
	    (ec->tx_coalesce_usecs > (TXGBE_MAX_EITR >> 2)))
		return -EINVAL;

	if (ec->use_adaptive_tx_coalesce)
		return -EINVAL;

	if (ec->use_adaptive_rx_coalesce) {
		adapter->rx_itr_setting = 1;
		return 0;
	} else {
		/* restore to default rxusecs value when adaptive itr turn off */
		/* user shall turn off adaptive itr and set user-defined rx usecs value
		 * in two cmds separately.
		 */
		if (adapter->rx_itr_setting == 1) {
			adapter->rx_itr_setting = TXGBE_20K_ITR;
			ec->rx_coalesce_usecs = adapter->rx_itr_setting >> 2;
		}
	}

	if (ec->rx_coalesce_usecs > 1)
		adapter->rx_itr_setting = ec->rx_coalesce_usecs << 2;
	else
		adapter->rx_itr_setting = ec->rx_coalesce_usecs;

	if (adapter->rx_itr_setting == 1)
		rx_itr_param = TXGBE_20K_ITR;
	else
		rx_itr_param = adapter->rx_itr_setting;

	if (ec->tx_coalesce_usecs > 1)
		adapter->tx_itr_setting = ec->tx_coalesce_usecs << 2;
	else
		adapter->tx_itr_setting = ec->tx_coalesce_usecs;

	if (adapter->tx_itr_setting == 1)
		tx_itr_param = TXGBE_12K_ITR;
	else
		tx_itr_param = adapter->tx_itr_setting;

	/* mixed Rx/Tx */
	if (adapter->q_vector[0]->tx.count && adapter->q_vector[0]->rx.count)
		adapter->tx_itr_setting = adapter->rx_itr_setting;

	/* detect ITR changes that require update of TXDCTL.WTHRESH */
	if ((adapter->tx_itr_setting != 1) &&
	    (adapter->tx_itr_setting < TXGBE_100K_ITR)) {
		if ((tx_itr_prev == 1) ||
		    (tx_itr_prev >= TXGBE_100K_ITR))
			need_reset = true;
	} else {
		if ((tx_itr_prev != 1) &&
		    (tx_itr_prev < TXGBE_100K_ITR))
			need_reset = true;
	}

	/* check the old value and enable RSC if necessary */
	need_reset |= txgbe_update_rsc(adapter);

	if (adapter->hw.mac.dmac_config.watchdog_timer &&
	    (!adapter->rx_itr_setting && !adapter->tx_itr_setting)) {
		e_info(probe,
		       "Disabling DMA coalescing because interrupt throttling "
		       "is disabled\n");
		adapter->hw.mac.dmac_config.watchdog_timer = 0;
		TCALL(hw, mac.ops.dmac_config);
	}

	for (i = 0; i < adapter->num_q_vectors; i++) {
		q_vector = adapter->q_vector[i];
		q_vector->tx.work_limit = adapter->tx_work_limit;
		q_vector->rx.work_limit = adapter->rx_work_limit;
		if (q_vector->tx.count && !q_vector->rx.count)
			/* tx only */
			q_vector->itr = tx_itr_param;
		else
			/* rx only or mixed */
			q_vector->itr = rx_itr_param;
		txgbe_write_eitr(q_vector);
	}

	/*
	 * do reset here at the end to make sure EITR==0 case is handled
	 * correctly w.r.t stopping tx, and changing TXDCTL.WTHRESH settings
	 * also locks in RSC enable/disable which requires reset
	 */
	if (need_reset)
		txgbe_do_reset(netdev);

	return 0;
}

#ifndef HAVE_NDO_SET_FEATURES
static u32 txgbe_get_rx_csum(struct net_device *netdev)
{
	return !!(netdev->features & NETIF_F_RXCSUM);
}

static int txgbe_set_rx_csum(struct net_device *netdev, u32 data)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	bool need_reset = false;

	if (data)
		netdev->features |= NETIF_F_RXCSUM;
	else
		netdev->features &= ~NETIF_F_RXCSUM;

	/* LRO and RSC both depend on RX checksum to function */
	if (!data && (netdev->features & NETIF_F_LRO)) {
		netdev->features &= ~NETIF_F_LRO;

		if (adapter->flags2 & TXGBE_FLAG2_RSC_ENABLED) {
			adapter->flags2 &= ~TXGBE_FLAG2_RSC_ENABLED;
			need_reset = true;
		}
	}

#ifdef HAVE_VXLAN_RX_OFFLOAD
	if (adapter->flags & TXGBE_FLAG_VXLAN_OFFLOAD_CAPABLE && data) {
		netdev->hw_enc_features |= NETIF_F_RXCSUM |
					   NETIF_F_IP_CSUM |
					   NETIF_F_IPV6_CSUM;
		if (!need_reset)
			adapter->flags2 |= TXGBE_FLAG2_VXLAN_REREG_NEEDED;
	} else {
		netdev->hw_enc_features &= ~(NETIF_F_RXCSUM |
					     NETIF_F_IP_CSUM |
					     NETIF_F_IPV6_CSUM);
		txgbe_clear_vxlan_port(adapter);
	}
#endif /* HAVE_VXLAN_RX_OFFLOAD */

	if (need_reset)
		txgbe_do_reset(netdev);

	return 0;
}

static int txgbe_set_tx_csum(struct net_device *netdev, u32 data)
{
#ifdef NETIF_F_IPV6_CSUM
	u32 feature_list = NETIF_F_IP_CSUM | NETIF_F_IPV6_CSUM;
#else
	u32 feature_list = NETIF_F_IP_CSUM;
#endif


#ifdef HAVE_ENCAP_TSO_OFFLOAD
	if (data)
		netdev->hw_enc_features |= NETIF_F_GSO_UDP_TUNNEL;
	else
		netdev->hw_enc_features &= ~NETIF_F_GSO_UDP_TUNNEL;
	feature_list |= NETIF_F_GSO_UDP_TUNNEL;
#endif /* HAVE_ENCAP_TSO_OFFLOAD */
	feature_list |= NETIF_F_SCTP_CSUM;


	if (data)
		netdev->features |= feature_list;
	else
		netdev->features &= ~feature_list;

	return 0;
}

#ifdef NETIF_F_TSO
static int txgbe_set_tso(struct net_device *netdev, u32 data)
{
#ifdef NETIF_F_TSO6
	u32 feature_list = NETIF_F_TSO | NETIF_F_TSO6;
#else
	u32 feature_list = NETIF_F_TSO;
#endif

	if (data)
		netdev->features |= feature_list;
	else
		netdev->features &= ~feature_list;

#ifndef HAVE_NETDEV_VLAN_FEATURES
	if (!data) {
		struct txgbe_adapter *adapter = netdev_priv(netdev);
		struct net_device *v_netdev;
		int i;

		/* disable TSO on all VLANs if they're present */
		if (!adapter->vlgrp)
			goto tso_out;

		for (i = 0; i < VLAN_GROUP_ARRAY_LEN; i++) {
			v_netdev = vlan_group_get_device(adapter->vlgrp, i);
			if (!v_netdev)
				continue;

			v_netdev->features &= ~feature_list;
			vlan_group_set_device(adapter->vlgrp, i, v_netdev);
		}
	}

tso_out:

#endif /* HAVE_NETDEV_VLAN_FEATURES */
	return 0;
}

#endif /* NETIF_F_TSO */
#ifdef ETHTOOL_GFLAGS
static int txgbe_set_flags(struct net_device *netdev, u32 data)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	u32 supported_flags = ETH_FLAG_RXVLAN | ETH_FLAG_TXVLAN;
	u32 changed = netdev->features ^ data;
	bool need_reset = false;
	int rc;

#ifndef HAVE_VLAN_RX_REGISTER
	if ((adapter->flags & TXGBE_FLAG_DCB_ENABLED) &&
	    !(data & ETH_FLAG_RXVLAN))
		return -EINVAL;

#endif
#ifdef TXGBE_NO_LRO
	if (adapter->flags2 & TXGBE_FLAG2_RSC_CAPABLE)
#endif
		supported_flags |= ETH_FLAG_LRO;

#ifdef ETHTOOL_GRXRINGS

	supported_flags |= ETH_FLAG_NTUPLE;


#endif
#ifdef NETIF_F_RXHASH
	supported_flags |= ETH_FLAG_RXHASH;

#endif
	rc = ethtool_op_set_flags(netdev, data, supported_flags);
	if (rc)
		return rc;

#ifndef HAVE_VLAN_RX_REGISTER
	if (changed & ETH_FLAG_RXVLAN)
		txgbe_vlan_mode(netdev, netdev->features);

#endif

#ifdef HAVE_VXLAN_CHECKS
	if (adapter->flags & TXGBE_FLAG_VXLAN_OFFLOAD_CAPABLE &&
	    netdev->features & NETIF_F_RXCSUM) {
		vxlan_get_rx_port(netdev);
	else
		txgbe_clear_vxlan_port(adapter);
	}
#endif /* HAVE_VXLAN_RX_OFFLOAD */

	/* if state changes we need to update adapter->flags and reset */
	if (!(netdev->features & NETIF_F_LRO)) {
		if (adapter->flags2 & TXGBE_FLAG2_RSC_ENABLED)
			need_reset = true;
		adapter->flags2 &= ~TXGBE_FLAG2_RSC_ENABLED;
	} else if ((adapter->flags2 & TXGBE_FLAG2_RSC_CAPABLE) &&
		   !(adapter->flags2 & TXGBE_FLAG2_RSC_ENABLED)) {
		if (adapter->rx_itr_setting == 1 ||
		    adapter->rx_itr_setting > TXGBE_MIN_RSC_ITR) {
			adapter->flags2 |= TXGBE_FLAG2_RSC_ENABLED;
			need_reset = true;
		} else if (changed & ETH_FLAG_LRO) {
#ifdef TXGBE_NO_LRO
			e_info(probe, "rx-usecs set too low, "
			       "disabling RSC\n");
#else
			e_info(probe, "rx-usecs set too low, "
			       "falling back to software LRO\n");
#endif
		}
	}

#ifdef ETHTOOL_GRXRINGS
	/*
	 * Check if Flow Director n-tuple support was enabled or disabled.  If
	 * the state changed, we need to reset.
	 */
	switch (netdev->features & NETIF_F_NTUPLE) {
	case NETIF_F_NTUPLE:
		/* turn off ATR, enable perfect filters and reset */
		if (!(adapter->flags & TXGBE_FLAG_FDIR_PERFECT_CAPABLE))
			need_reset = true;

		adapter->flags &= ~TXGBE_FLAG_FDIR_HASH_CAPABLE;
		adapter->flags |= TXGBE_FLAG_FDIR_PERFECT_CAPABLE;
		break;
	default:
		/* turn off perfect filters, enable ATR and reset */
		if (adapter->flags & TXGBE_FLAG_FDIR_PERFECT_CAPABLE)
			need_reset = true;

		adapter->flags &= ~TXGBE_FLAG_FDIR_PERFECT_CAPABLE;

		/* We cannot enable ATR if VMDq is enabled */
		if (adapter->flags & TXGBE_FLAG_VMDQ_ENABLED)
			break;

		/* We cannot enable ATR if we have 2 or more traffic classes */
		if (netdev_get_num_tc(netdev) > 1)
			break;

		/* We cannot enable ATR if RSS is disabled */
		if (adapter->ring_feature[RING_F_RSS].limit <= 1)
			break;

		/* A sample rate of 0 indicates ATR disabled */
		if (!adapter->atr_sample_rate)
			break;

		adapter->flags |= TXGBE_FLAG_FDIR_HASH_CAPABLE;
		break;
	}

#endif /* ETHTOOL_GRXRINGS */
	if (need_reset)
		txgbe_do_reset(netdev);

	return 0;
}

#endif /* ETHTOOL_GFLAGS */
#endif /* HAVE_NDO_SET_FEATURES */
#ifdef ETHTOOL_GRXRINGS
static int txgbe_match_etype_entry(struct txgbe_adapter *adapter, u16 sw_idx)
{
	struct txgbe_etype_filter_info *ef_info = &adapter->etype_filter_info;
	int i;

	for (i = 0; i < TXGBE_MAX_PSR_ETYPE_SWC_FILTERS; i++) {
		if (ef_info->etype_filters[i].rule_idx == sw_idx)
			break;
	}

	return i;
}

static int txgbe_get_etype_rule(struct txgbe_adapter *adapter,
				struct ethtool_rx_flow_spec *fsp, int ef_idx)
{
	struct txgbe_etype_filter_info *ef_info = &adapter->etype_filter_info;
	u8 mask[6] = {0, 0, 0, 0, 0, 0};
	u8 mac[6] = {0, 0, 0, 0, 0, 0};

	fsp->flow_type = ETHER_FLOW;
	ether_addr_copy(fsp->h_u.ether_spec.h_dest, mac);
	ether_addr_copy(fsp->m_u.ether_spec.h_dest, mask);
	ether_addr_copy(fsp->h_u.ether_spec.h_source, mac);
	ether_addr_copy(fsp->m_u.ether_spec.h_source, mask);
	fsp->h_u.ether_spec.h_proto = htons(ef_info->etype_filters[ef_idx].ethertype);
	fsp->m_u.ether_spec.h_proto = 0xFFFF;
	fsp->ring_cookie = ef_info->etype_filters[ef_idx].action;

	return 0;
}

static int txgbe_get_ethtool_fdir_entry(struct txgbe_adapter *adapter,
					struct ethtool_rxnfc *cmd)
{
	union txgbe_atr_input *mask = &adapter->fdir_mask;
	struct ethtool_rx_flow_spec *fsp =
		(struct ethtool_rx_flow_spec *)&cmd->fs;
	struct hlist_node *node;
	struct txgbe_fdir_filter *rule = NULL;

	if (adapter->etype_filter_info.count > 0) {
		int ef_idx;

		ef_idx = txgbe_match_etype_entry(adapter, fsp->location);
		if (ef_idx < TXGBE_MAX_PSR_ETYPE_SWC_FILTERS)
			return txgbe_get_etype_rule(adapter, fsp, ef_idx);
	}

	/* report total rule count */
	cmd->data = (1024 << adapter->fdir_pballoc) - 2;

	hlist_for_each_entry_safe(rule, node,
				  &adapter->fdir_filter_list, fdir_node) {
		if (fsp->location <= rule->sw_idx)
			break;
	}

	if (!rule || fsp->location != rule->sw_idx)
		return -EINVAL;

	/* fill out the flow spec entry */

	/* set flow type field */
	switch (rule->filter.formatted.flow_type) {
	case TXGBE_ATR_FLOW_TYPE_TCPV4:
		fsp->flow_type = TCP_V4_FLOW;
		break;
	case TXGBE_ATR_FLOW_TYPE_UDPV4:
		fsp->flow_type = UDP_V4_FLOW;
		break;
	case TXGBE_ATR_FLOW_TYPE_SCTPV4:
		fsp->flow_type = SCTP_V4_FLOW;
		break;
	case TXGBE_ATR_FLOW_TYPE_IPV4:
		fsp->flow_type = IP_USER_FLOW;
		fsp->h_u.usr_ip4_spec.ip_ver = ETH_RX_NFC_IP4;
		fsp->h_u.usr_ip4_spec.proto = 0;
		fsp->m_u.usr_ip4_spec.proto = 0;
		break;
	default:
		return -EINVAL;
	}

	fsp->h_u.tcp_ip4_spec.psrc = rule->filter.formatted.src_port;
	fsp->m_u.tcp_ip4_spec.psrc = mask->formatted.src_port;
	fsp->h_u.tcp_ip4_spec.pdst = rule->filter.formatted.dst_port;
	fsp->m_u.tcp_ip4_spec.pdst = mask->formatted.dst_port;
	fsp->h_u.tcp_ip4_spec.ip4src = rule->filter.formatted.src_ip[0];
	fsp->m_u.tcp_ip4_spec.ip4src = mask->formatted.src_ip[0];
	fsp->h_u.tcp_ip4_spec.ip4dst = rule->filter.formatted.dst_ip[0];
	fsp->m_u.tcp_ip4_spec.ip4dst = mask->formatted.dst_ip[0];
	fsp->h_ext.vlan_etype = rule->filter.formatted.flex_bytes;
	fsp->m_ext.vlan_etype = mask->formatted.flex_bytes;
	fsp->h_ext.data[1] = htonl(rule->filter.formatted.vm_pool);
	fsp->m_ext.data[1] = htonl(mask->formatted.vm_pool);
	fsp->flow_type |= FLOW_EXT;

	/* record action */
	if (rule->action == TXGBE_RDB_FDIR_DROP_QUEUE)
		fsp->ring_cookie = RX_CLS_FLOW_DISC;
	else
		fsp->ring_cookie = rule->action;

	return 0;
}

static int txgbe_get_ethtool_fdir_all(struct txgbe_adapter *adapter,
				      struct ethtool_rxnfc *cmd,
				      u32 *rule_locs)
{
	struct txgbe_etype_filter_info *ef_info = &adapter->etype_filter_info;
	struct hlist_node *node;
	struct txgbe_fdir_filter *rule;
	int cnt = 0, i;

	/* report total rule count */
	cmd->data = (1024 << adapter->fdir_pballoc) - 2;

	hlist_for_each_entry_safe(rule, node,
				  &adapter->fdir_filter_list, fdir_node) {
		if (cnt == cmd->rule_cnt)
			return -EMSGSIZE;
		rule_locs[cnt] = rule->sw_idx;
		cnt++;
	}

	for (i = 0; i < TXGBE_MAX_PSR_ETYPE_SWC_FILTERS; i++) {
		if (ef_info->ethertype_mask & (1 << i)) {
			rule_locs[cnt] = ef_info->etype_filters[i].rule_idx;
			cnt++;
		}
	}

	cmd->rule_cnt = cnt;

	return 0;
}

static int txgbe_get_rss_hash_opts(struct txgbe_adapter *adapter,
				   struct ethtool_rxnfc *cmd)
{
	cmd->data = 0;

	/* Report default options for RSS on txgbe */
	switch (cmd->flow_type) {
	case TCP_V4_FLOW:
		cmd->data |= RXH_L4_B_0_1 | RXH_L4_B_2_3;
		fallthrough;
	case UDP_V4_FLOW:
		if (adapter->flags2 & TXGBE_FLAG2_RSS_FIELD_IPV4_UDP)
			cmd->data |= RXH_L4_B_0_1 | RXH_L4_B_2_3;
		fallthrough;
	case SCTP_V4_FLOW:
	case AH_ESP_V4_FLOW:
	case AH_V4_FLOW:
	case ESP_V4_FLOW:
	case IPV4_FLOW:
		cmd->data |= RXH_IP_SRC | RXH_IP_DST;
		break;
	case TCP_V6_FLOW:
		cmd->data |= RXH_L4_B_0_1 | RXH_L4_B_2_3;
		fallthrough;
	case UDP_V6_FLOW:
		if (adapter->flags2 & TXGBE_FLAG2_RSS_FIELD_IPV6_UDP)
			cmd->data |= RXH_L4_B_0_1 | RXH_L4_B_2_3;
		fallthrough;
	case SCTP_V6_FLOW:
	case AH_ESP_V6_FLOW:
	case AH_V6_FLOW:
	case ESP_V6_FLOW:
	case IPV6_FLOW:
		cmd->data |= RXH_IP_SRC | RXH_IP_DST;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static int txgbe_get_rxnfc(struct net_device *dev, struct ethtool_rxnfc *cmd,
#ifdef HAVE_ETHTOOL_GET_RXNFC_VOID_RULE_LOCS
			   void *rule_locs)
#else
			   u32 *rule_locs)
#endif
{
	struct txgbe_adapter *adapter = netdev_priv(dev);
	int ret = -EOPNOTSUPP;

	switch (cmd->cmd) {
	case ETHTOOL_GRXRINGS:
		cmd->data = adapter->num_rx_queues;
		ret = 0;
		break;
	case ETHTOOL_GRXCLSRLCNT:
		cmd->rule_cnt = adapter->fdir_filter_count +
				adapter->etype_filter_info.count;
		ret = 0;
		break;
	case ETHTOOL_GRXCLSRULE:
		ret = txgbe_get_ethtool_fdir_entry(adapter, cmd);
		break;
	case ETHTOOL_GRXCLSRLALL:
		ret = txgbe_get_ethtool_fdir_all(adapter, cmd,
						 (u32 *)rule_locs);
		break;
	case ETHTOOL_GRXFH:
		ret = txgbe_get_rss_hash_opts(adapter, cmd);
		break;
	default:
		break;
	}

	return ret;
}

static int
txgbe_ethertype_filter_lookup(struct txgbe_etype_filter_info *ef_info,
			      u16 ethertype)
{
	int i;

	for (i = 0; i < TXGBE_MAX_PSR_ETYPE_SWC_FILTERS; i++) {
		if (ef_info->etype_filters[i].ethertype == ethertype &&
		    (ef_info->ethertype_mask & (1 << i)))
			return i;
	}
	return -1;
}

static int
txgbe_ethertype_filter_insert(struct txgbe_etype_filter_info *ef_info,
			      struct txgbe_ethertype_filter *etype_filter)
{
	int i;

	for (i = 0; i < TXGBE_MAX_PSR_ETYPE_SWC_FILTERS; i++) {
		if (ef_info->ethertype_mask & (1 << i)) {
			continue;
		}
		ef_info->ethertype_mask |= 1 << i;
		ef_info->etype_filters[i].ethertype = etype_filter->ethertype;
		ef_info->etype_filters[i].etqf = etype_filter->etqf;
		ef_info->etype_filters[i].etqs = etype_filter->etqs;
		ef_info->etype_filters[i].rule_idx = etype_filter->rule_idx;
		ef_info->etype_filters[i].action = etype_filter->action;
		break;
	}

	return (i < TXGBE_MAX_PSR_ETYPE_SWC_FILTERS ? i : -1);
}

static int txgbe_add_ethertype_filter(struct txgbe_adapter *adapter,
				      struct ethtool_rx_flow_spec *fsp)
{
	struct txgbe_etype_filter_info *ef_info = &adapter->etype_filter_info;
	struct txgbe_ethertype_filter etype_filter;
	struct txgbe_hw *hw = &adapter->hw;
	u16 ethertype;
	u32 etqf = 0;
	u32 etqs = 0;
	u8 queue, vf;
	u32 ring;
	int ret;

	ethertype = ntohs(fsp->h_u.ether_spec.h_proto);
	if (!ethertype) {
		e_err(drv, "protocol number is missing for ethertype filter\n");
		return -EINVAL;
	}
	if (ethertype == ETH_P_IP || ethertype == ETH_P_IPV6) {
		e_err(drv, "unsupported ether_type(0x%04x) in ethertype filter\n",
			ethertype);
		return -EINVAL;
	}

	ret = txgbe_ethertype_filter_lookup(ef_info, ethertype);
	if (ret >= 0) {
		e_err(drv, "ethertype (0x%04x) filter exists.", ethertype);
		return -EEXIST;
	}

	/* ring_cookie is a masked into a set of queues and txgbe pools */
	if (fsp->ring_cookie == RX_CLS_FLOW_DISC) {
		e_err(drv, "drop option is unsupported.");
		return -EINVAL;
	}

	ring = ethtool_get_flow_spec_ring(fsp->ring_cookie);
	vf = ethtool_get_flow_spec_ring_vf(fsp->ring_cookie);
	if (!vf && ring >= adapter->num_rx_queues)
		return -EINVAL;
	else if (vf && ((vf > adapter->num_vfs) ||
			ring >= adapter->num_rx_queues_per_pool))
		return -EINVAL;

	/* Map the ring onto the absolute queue index */
	if (!vf)
		queue = adapter->rx_ring[ring]->reg_idx;
	else
		queue = ((vf - 1) * adapter->num_rx_queues_per_pool) + ring;

	etqs |= queue << TXGBE_RDB_ETYPE_CLS_RX_QUEUE_SHIFT;
	etqs |= TXGBE_RDB_ETYPE_CLS_QUEUE_EN;
	etqf = TXGBE_PSR_ETYPE_SWC_FILTER_EN | ethertype;
	if (adapter->num_vfs) {
		u8 pool;

		if (!vf)
			pool = adapter->num_vfs;
		else
			pool = vf - 1;

		etqf |= TXGBE_PSR_ETYPE_SWC_POOL_ENABLE;
		etqf |= pool << TXGBE_PSR_ETYPE_SWC_POOL_SHIFT;
	}

	etype_filter.ethertype = ethertype;
	etype_filter.etqf = etqf;
	etype_filter.etqs = etqs;
	etype_filter.rule_idx = fsp->location;
	etype_filter.action = fsp->ring_cookie;
	ret = txgbe_ethertype_filter_insert(ef_info, &etype_filter);
	if (ret < 0) {
		e_err(drv, "ethertype filters are full.");
		return -ENOSPC;
	}

	wr32(hw, TXGBE_PSR_ETYPE_SWC(ret), etqf);
	wr32(hw, TXGBE_RDB_ETYPE_CLS(ret), etqs);
	TXGBE_WRITE_FLUSH(hw);

	ef_info->count++;

	return 0;
}

static int txgbe_del_ethertype_filter(struct txgbe_adapter *adapter, u16 sw_idx)
{
	struct txgbe_etype_filter_info *ef_info = &adapter->etype_filter_info;
	struct txgbe_hw *hw = &adapter->hw;
	u16 ethertype;
	int idx;

	idx = txgbe_match_etype_entry(adapter, sw_idx);
	if (idx == TXGBE_MAX_PSR_ETYPE_SWC_FILTERS)
		return -EINVAL;

	ethertype = ef_info->etype_filters[idx].ethertype;
	if (!ethertype) {
		e_err(drv, "ethertype filter doesn't exist.");
		return -ENOENT;
	}

	ef_info->ethertype_mask &= ~(1 << idx);
	ef_info->etype_filters[idx].ethertype = 0;
	ef_info->etype_filters[idx].etqf = 0;
	ef_info->etype_filters[idx].etqs = 0;
	ef_info->etype_filters[idx].etqs = FALSE;
	ef_info->etype_filters[idx].rule_idx = 0;

	wr32(hw, TXGBE_PSR_ETYPE_SWC(idx), 0);
	wr32(hw, TXGBE_RDB_ETYPE_CLS(idx), 0);
	TXGBE_WRITE_FLUSH(hw);

	ef_info->count--;

	return 0;

}

static int txgbe_update_ethtool_fdir_entry(struct txgbe_adapter *adapter,
					   struct txgbe_fdir_filter *input,
					   u16 sw_idx)
{
	struct txgbe_hw *hw = &adapter->hw;
	struct hlist_node *node, *parent;
	struct txgbe_fdir_filter *rule;
	bool deleted = false;
	s32 err;

	parent = NULL;
	rule = NULL;

	hlist_for_each_entry_safe(rule, node,
				  &adapter->fdir_filter_list, fdir_node) {
		/* hash found, or no matching entry */
		if (rule->sw_idx >= sw_idx)
			break;
		parent = node;
	}

	/* if there is an old rule occupying our place remove it */
	if (rule && (rule->sw_idx == sw_idx)) {
		/* hardware filters are only configured when interface is up,
		 * and we should not issue filter commands while the interface
		 * is down
		 */
		if (netif_running(adapter->netdev) &&
		    (!input || (rule->filter.formatted.bkt_hash !=
				input->filter.formatted.bkt_hash))) {
			err = txgbe_fdir_erase_perfect_filter(hw,
								&rule->filter,
								sw_idx);
			if (err)
				return -EINVAL;
		}

		hlist_del(&rule->fdir_node);
		kfree(rule);
		adapter->fdir_filter_count--;
		deleted = true;
	}

	/* If we weren't given an input, then this was a request to delete a
	 * filter. We should return -EINVAL if the filter wasn't found, but
	 * return 0 if the rule was successfully deleted.
	 */
	if (!input)
		return deleted ? 0 : -EINVAL;

	/* initialize node and set software index */
	INIT_HLIST_NODE(&input->fdir_node);

	/* add filter to the list */
	if (parent)
		hlist_add_behind(&input->fdir_node, parent);
	else
		hlist_add_head(&input->fdir_node,
			       &adapter->fdir_filter_list);

	/* update counts */
	adapter->fdir_filter_count++;

	return 0;
}

static int txgbe_flowspec_to_flow_type(struct ethtool_rx_flow_spec *fsp,
				       u8 *flow_type)
{
	switch (fsp->flow_type & ~FLOW_EXT) {
	case TCP_V4_FLOW:
		*flow_type = TXGBE_ATR_FLOW_TYPE_TCPV4;
		break;
	case UDP_V4_FLOW:
		*flow_type = TXGBE_ATR_FLOW_TYPE_UDPV4;
		break;
	case SCTP_V4_FLOW:
		*flow_type = TXGBE_ATR_FLOW_TYPE_SCTPV4;
		break;
	case IP_USER_FLOW:
		switch (fsp->h_u.usr_ip4_spec.proto) {
		case IPPROTO_TCP:
			*flow_type = TXGBE_ATR_FLOW_TYPE_TCPV4;
			break;
		case IPPROTO_UDP:
			*flow_type = TXGBE_ATR_FLOW_TYPE_UDPV4;
			break;
		case IPPROTO_SCTP:
			*flow_type = TXGBE_ATR_FLOW_TYPE_SCTPV4;
			break;
		case 0:
			if (!fsp->m_u.usr_ip4_spec.proto) {
				*flow_type = TXGBE_ATR_FLOW_TYPE_IPV4;
				break;
			}
			fallthrough;
		default:
			return 0;
		}
		break;
	default:
		return 0;
	}

	return 1;
}

 static bool txgbe_match_ethtool_fdir_entry(struct txgbe_adapter *adapter,
						  struct txgbe_fdir_filter *input)
{
	struct hlist_node *node2;
	struct txgbe_fdir_filter *rule = NULL;

	hlist_for_each_entry_safe(rule, node2,
					 &adapter->fdir_filter_list, fdir_node) {
		if (rule->filter.formatted.bkt_hash ==
			input->filter.formatted.bkt_hash &&
			rule->action == input->action) {
			e_info(drv, "FDIR entry already exist\n");
			return true;
		}
	}
	return false;
}

static int txgbe_add_ethtool_fdir_entry(struct txgbe_adapter *adapter,
					struct ethtool_rxnfc *cmd)
{
	struct ethtool_rx_flow_spec *fsp =
		(struct ethtool_rx_flow_spec *)&cmd->fs;
	struct txgbe_hw *hw = &adapter->hw;
	struct txgbe_fdir_filter *input;
	union txgbe_atr_input mask;
	u8 queue;
	int err;
	u16 ptype = 0;

	if ((fsp->flow_type & ~FLOW_EXT) == ETHER_FLOW)
		return txgbe_add_ethertype_filter(adapter, fsp);

	if (!(adapter->flags & TXGBE_FLAG_FDIR_PERFECT_CAPABLE))
		return -EOPNOTSUPP;

	/* ring_cookie is a masked into a set of queues and txgbe pools or
	 * we use drop index
	 */
	if (fsp->ring_cookie == RX_CLS_FLOW_DISC) {
		queue = TXGBE_RDB_FDIR_DROP_QUEUE;
	} else {
		u32 ring = ethtool_get_flow_spec_ring(fsp->ring_cookie);
		u8 vf = ethtool_get_flow_spec_ring_vf(fsp->ring_cookie);

		if (!vf && ring >= adapter->num_rx_queues)
			return -EINVAL;
		else if (vf &&
			 ((vf > adapter->num_vfs) ||
			   ring >= adapter->num_rx_queues_per_pool))
			return -EINVAL;

		/* Map the ring onto the absolute queue index */
		if (!vf)
			queue = adapter->rx_ring[ring]->reg_idx;
		else
			queue = ((vf - 1) *
				adapter->num_rx_queues_per_pool) + ring;
	}


	/* Don't allow indexes to exist outside of available space */
	if (fsp->location >= ((1024 << adapter->fdir_pballoc) - 2)) {
		e_err(drv, "Location out of range\n");
		return -EINVAL;
	}

	input = kzalloc(sizeof(*input), GFP_ATOMIC);
	if (!input)
		return -ENOMEM;

	memset(&mask, 0, sizeof(union txgbe_atr_input));

	/* set SW index */
	input->sw_idx = fsp->location;

	/* record flow type */
	if (!txgbe_flowspec_to_flow_type(fsp,
					 &input->filter.formatted.flow_type)) {
		e_err(drv, "Unrecognized flow type\n");
		goto err_out;
	}

	mask.formatted.flow_type = TXGBE_ATR_L4TYPE_IPV6_MASK |
				   TXGBE_ATR_L4TYPE_MASK;

	if (input->filter.formatted.flow_type == TXGBE_ATR_FLOW_TYPE_IPV4)
		mask.formatted.flow_type &= TXGBE_ATR_L4TYPE_IPV6_MASK;

	/* Copy input into formatted structures */
	input->filter.formatted.src_ip[0] = fsp->h_u.tcp_ip4_spec.ip4src;
	mask.formatted.src_ip[0] = fsp->m_u.tcp_ip4_spec.ip4src;
	input->filter.formatted.dst_ip[0] = fsp->h_u.tcp_ip4_spec.ip4dst;
	mask.formatted.dst_ip[0] = fsp->m_u.tcp_ip4_spec.ip4dst;
	input->filter.formatted.src_port = fsp->h_u.tcp_ip4_spec.psrc;
	mask.formatted.src_port = fsp->m_u.tcp_ip4_spec.psrc;
	input->filter.formatted.dst_port = fsp->h_u.tcp_ip4_spec.pdst;
	mask.formatted.dst_port = fsp->m_u.tcp_ip4_spec.pdst;

	if (fsp->flow_type & FLOW_EXT) {
		input->filter.formatted.vm_pool =
				(unsigned char)ntohl(fsp->h_ext.data[1]);
		mask.formatted.vm_pool =
				(unsigned char)ntohl(fsp->m_ext.data[1]);
		input->filter.formatted.flex_bytes =
						fsp->h_ext.vlan_etype;
		mask.formatted.flex_bytes = fsp->m_ext.vlan_etype;
#ifdef FIXED
		/* need fix */
		input->filter.formatted.tunnel_type =
				(unsigned char)ntohl(fsp->h_ext.data[0]);
		mask.formatted.tunnel_type =
				(unsigned char)ntohl(fsp->m_ext.data[0]);
#endif
	}

	switch (input->filter.formatted.flow_type) {
	case TXGBE_ATR_FLOW_TYPE_TCPV4:
		ptype = TXGBE_PTYPE_L2_IPV4_TCP;
		break;
	case TXGBE_ATR_FLOW_TYPE_UDPV4:
		ptype = TXGBE_PTYPE_L2_IPV4_UDP;
		break;
	case TXGBE_ATR_FLOW_TYPE_SCTPV4:
		ptype = TXGBE_PTYPE_L2_IPV4_SCTP;
		break;
	case TXGBE_ATR_FLOW_TYPE_IPV4:
		ptype = TXGBE_PTYPE_L2_IPV4;
		break;
	case TXGBE_ATR_FLOW_TYPE_TCPV6:
		ptype = TXGBE_PTYPE_L2_IPV6_TCP;
		break;
	case TXGBE_ATR_FLOW_TYPE_UDPV6:
		ptype = TXGBE_PTYPE_L2_IPV6_UDP;
		break;
	case TXGBE_ATR_FLOW_TYPE_SCTPV6:
		ptype = TXGBE_PTYPE_L2_IPV6_SCTP;
		break;
	case TXGBE_ATR_FLOW_TYPE_IPV6:
		ptype = TXGBE_PTYPE_L2_IPV6;
		break;
	default:
		break;
	}

	input->filter.formatted.vlan_id = htons(ptype);
	if (mask.formatted.flow_type & TXGBE_ATR_L4TYPE_MASK)
		mask.formatted.vlan_id = 0xFFFF;
	else
		mask.formatted.vlan_id = htons(0xFFF8);

	/* determine if we need to drop or route the packet */
	if (fsp->ring_cookie == RX_CLS_FLOW_DISC)
		input->action = TXGBE_RDB_FDIR_DROP_QUEUE;
	else
		input->action = fsp->ring_cookie;

	spin_lock(&adapter->fdir_perfect_lock);

	if (hlist_empty(&adapter->fdir_filter_list)) {
		/* save mask and program input mask into HW */
		memcpy(&adapter->fdir_mask, &mask, sizeof(mask));
		err = txgbe_fdir_set_input_mask(hw, &mask,
							 adapter->cloud_mode);
		if (err) {
			e_err(drv, "Error writing mask\n");
			goto err_out_w_lock;
		}
	} else if (memcmp(&adapter->fdir_mask, &mask, sizeof(mask))) {
		e_err(drv, "Hardware only supports one mask per port. To change"
		      "the mask you must first delete all the rules.\n");
		goto err_out_w_lock;
	}

	/* apply mask and compute/store hash */
	txgbe_atr_compute_perfect_hash(&input->filter, &mask);

	/* check if new entry does not exist on filter list */
	if (txgbe_match_ethtool_fdir_entry(adapter, input))
		goto err_out_w_lock;

	/* only program filters to hardware if the net device is running, as
	 * we store the filters in the Rx buffer which is not allocated when
	 * the device is down
	 */
	if (netif_running(adapter->netdev)) {
		err = txgbe_fdir_write_perfect_filter(hw,
				&input->filter, input->sw_idx,
				queue,
				adapter->cloud_mode);
		if (err)
			goto err_out_w_lock;
	}

	txgbe_update_ethtool_fdir_entry(adapter, input, input->sw_idx);

	spin_unlock(&adapter->fdir_perfect_lock);

	return err;
err_out_w_lock:
	spin_unlock(&adapter->fdir_perfect_lock);
err_out:
	kfree(input);
	return -EINVAL;
}

static int txgbe_del_ethtool_fdir_entry(struct txgbe_adapter *adapter,
					struct ethtool_rxnfc *cmd)
{
	struct ethtool_rx_flow_spec *fsp =
		(struct ethtool_rx_flow_spec *)&cmd->fs;
	int err;

	if (adapter->etype_filter_info.count > 0) {
		err = txgbe_del_ethertype_filter(adapter, fsp->location);
		if (!err)
			return 0;
	}

	spin_lock(&adapter->fdir_perfect_lock);
	err = txgbe_update_ethtool_fdir_entry(adapter, NULL, fsp->location);
	spin_unlock(&adapter->fdir_perfect_lock);

	return err;
}

#ifdef ETHTOOL_SRXNTUPLE
/*
 * We need to keep this around for kernels 2.6.33 - 2.6.39 in order to avoid
 * a null pointer dereference as it was assumend if the NETIF_F_NTUPLE flag
 * was defined that this function was present.
 */
static int txgbe_set_rx_ntuple(struct net_device __always_unused *dev,
			       struct ethtool_rx_ntuple __always_unused *cmd)
{
	return -EOPNOTSUPP;
}

#endif
#define UDP_RSS_FLAGS (TXGBE_FLAG2_RSS_FIELD_IPV4_UDP | \
		       TXGBE_FLAG2_RSS_FIELD_IPV6_UDP)
static int txgbe_set_rss_hash_opt(struct txgbe_adapter *adapter,
				  struct ethtool_rxnfc *nfc)
{
	u32 flags2 = adapter->flags2;

	/*
	 * RSS does not support anything other than hashing
	 * to queues on src and dst IPs and ports
	 */
	if (nfc->data & ~(RXH_IP_SRC | RXH_IP_DST |
			  RXH_L4_B_0_1 | RXH_L4_B_2_3))
		return -EINVAL;

	switch (nfc->flow_type) {
	case TCP_V4_FLOW:
	case TCP_V6_FLOW:
		if (!(nfc->data & RXH_IP_SRC) ||
		    !(nfc->data & RXH_IP_DST) ||
		    !(nfc->data & RXH_L4_B_0_1) ||
		    !(nfc->data & RXH_L4_B_2_3))
			return -EINVAL;
		break;
	case UDP_V4_FLOW:
		if (!(nfc->data & RXH_IP_SRC) ||
		    !(nfc->data & RXH_IP_DST))
			return -EINVAL;
		switch (nfc->data & (RXH_L4_B_0_1 | RXH_L4_B_2_3)) {
		case 0:
			flags2 &= ~TXGBE_FLAG2_RSS_FIELD_IPV4_UDP;
			break;
		case (RXH_L4_B_0_1 | RXH_L4_B_2_3):
			flags2 |= TXGBE_FLAG2_RSS_FIELD_IPV4_UDP;
			break;
		default:
			return -EINVAL;
		}
		break;
	case UDP_V6_FLOW:
		if (!(nfc->data & RXH_IP_SRC) ||
		    !(nfc->data & RXH_IP_DST))
			return -EINVAL;
		switch (nfc->data & (RXH_L4_B_0_1 | RXH_L4_B_2_3)) {
		case 0:
			flags2 &= ~TXGBE_FLAG2_RSS_FIELD_IPV6_UDP;
			break;
		case (RXH_L4_B_0_1 | RXH_L4_B_2_3):
			flags2 |= TXGBE_FLAG2_RSS_FIELD_IPV6_UDP;
			break;
		default:
			return -EINVAL;
		}
		break;
	case AH_ESP_V4_FLOW:
	case AH_V4_FLOW:
	case ESP_V4_FLOW:
	case SCTP_V4_FLOW:
	case AH_ESP_V6_FLOW:
	case AH_V6_FLOW:
	case ESP_V6_FLOW:
	case SCTP_V6_FLOW:
		if (!(nfc->data & RXH_IP_SRC) ||
		    !(nfc->data & RXH_IP_DST) ||
		    (nfc->data & RXH_L4_B_0_1) ||
		    (nfc->data & RXH_L4_B_2_3))
			return -EINVAL;
		break;
	default:
		return -EINVAL;
	}

	/* if we changed something we need to update flags */
	if (flags2 != adapter->flags2) {
		struct txgbe_hw *hw = &adapter->hw;
		u32 mrqc;

		mrqc = rd32(hw, TXGBE_RDB_RA_CTL);

		if ((flags2 & UDP_RSS_FLAGS) &&
		    !(adapter->flags2 & UDP_RSS_FLAGS))
			e_warn(drv, "enabling UDP RSS: fragmented packets"
			       " may arrive out of order to the stack above\n");

		adapter->flags2 = flags2;

		/* Perform hash on these packet types */
		mrqc |= TXGBE_RDB_RA_CTL_RSS_IPV4
		      | TXGBE_RDB_RA_CTL_RSS_IPV4_TCP
		      | TXGBE_RDB_RA_CTL_RSS_IPV6
		      | TXGBE_RDB_RA_CTL_RSS_IPV6_TCP;

		mrqc &= ~(TXGBE_RDB_RA_CTL_RSS_IPV4_UDP |
			  TXGBE_RDB_RA_CTL_RSS_IPV6_UDP);

		if (flags2 & TXGBE_FLAG2_RSS_FIELD_IPV4_UDP)
			mrqc |= TXGBE_RDB_RA_CTL_RSS_IPV4_UDP;

		if (flags2 & TXGBE_FLAG2_RSS_FIELD_IPV6_UDP)
			mrqc |= TXGBE_RDB_RA_CTL_RSS_IPV6_UDP;

		wr32(hw, TXGBE_RDB_RA_CTL, mrqc);
	}

	return 0;
}

static int txgbe_set_rxnfc(struct net_device *dev, struct ethtool_rxnfc *cmd)
{
	struct txgbe_adapter *adapter = netdev_priv(dev);
	int ret = -EOPNOTSUPP;

	switch (cmd->cmd) {
	case ETHTOOL_SRXCLSRLINS:
		ret = txgbe_add_ethtool_fdir_entry(adapter, cmd);
		break;
	case ETHTOOL_SRXCLSRLDEL:
		ret = txgbe_del_ethtool_fdir_entry(adapter, cmd);
		break;
	case ETHTOOL_SRXFH:
		ret = txgbe_set_rss_hash_opt(adapter, cmd);
		break;
	default:
		break;
	}

	return ret;
}

#if defined(ETHTOOL_GRSSH) && defined(ETHTOOL_SRSSH)
static int txgbe_rss_indir_tbl_max(struct txgbe_adapter *adapter)
{
	return 64;
}


static u32 txgbe_get_rxfh_key_size(struct net_device *netdev)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);

	return sizeof(adapter->rss_key);
}

static u32 txgbe_rss_indir_size(struct net_device *netdev)
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);

	return txgbe_rss_indir_tbl_entries(adapter);
}

static void txgbe_get_reta(struct txgbe_adapter *adapter, u32 *indir)
{
	int i, reta_size = txgbe_rss_indir_tbl_entries(adapter);

	for (i = 0; i < reta_size; i++)
		indir[i] = adapter->rss_indir_tbl[i];
}

#ifdef HAVE_ETHTOOL_RXFH_RXFHPARAMS
static int txgbe_get_rxfh(struct net_device *netdev,
			  struct ethtool_rxfh_param *rxfh)
#else
#ifdef HAVE_RXFH_HASHFUNC
static int txgbe_get_rxfh(struct net_device *netdev, u32 *indir, u8 *key,
			  u8 *hfunc)
#else /* HAVE_RXFH_HASHFUNC */
static int txgbe_get_rxfh(struct net_device *netdev, u32 *indir, u8 *key)
#endif /* HAVE_RXFH_HASHFUNC */
#endif /* HAVE_ETHTOOL_RXFH_RXFHPARAMS */
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
#ifdef HAVE_ETHTOOL_RXFH_RXFHPARAMS
	u8 *key = rxfh->key;
	u32 *indir = rxfh->indir;
#else
#ifdef HAVE_RXFH_HASHFUNC
	if (hfunc)
		*hfunc = ETH_RSS_HASH_TOP;
#endif
#endif

	if (indir)
		txgbe_get_reta(adapter, indir);
	if (key)
		memcpy(key, adapter->rss_key, txgbe_get_rxfh_key_size(netdev));

	return 0;
}

#ifdef HAVE_ETHTOOL_RXFH_RXFHPARAMS
static int txgbe_set_rxfh(struct net_device *netdev,
			  struct ethtool_rxfh_param *rxfh,
			  struct netlink_ext_ack *extack)
#else
#ifdef HAVE_RXFH_HASHFUNC
static int txgbe_set_rxfh(struct net_device *netdev, const u32 *indir,
			  const u8 *key, const u8 hfunc)
#else
#ifdef HAVE_RXFH_NONCONST
static int txgbe_set_rxfh(struct net_device *netdev, u32 *indir, u8 *key)
#else /* HAVE_RXFH_NONCONST */
static int txgbe_set_rxfh(struct net_device *netdev, const u32 *indir,
			  const u8 *key)
#endif /* HAVE_RXFH_NONCONST */
#endif /* HAVE_RXFH_HASHFUNC */
#endif /* HAVE_ETHTOOL_RXFH_RXFHPARAMS */
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	int i;
	u32 reta_entries = txgbe_rss_indir_tbl_entries(adapter);
	struct txgbe_hw *hw = &adapter->hw;
#ifdef HAVE_ETHTOOL_RXFH_RXFHPARAMS
	u8 hfunc = rxfh->hfunc;
	u8 *key = rxfh->key;
	u32 *indir = rxfh->indir;
#endif

#if (defined(HAVE_RXFH_HASHFUNC) || defined(HAVE_ETHTOOL_RXFH_RXFHPARAMS))
	if (hfunc)
		return -EINVAL;
#endif

	/* Fill out the redirection table */
	if (indir) {
		int max_queues = min_t(int, adapter->num_rx_queues,
				       txgbe_rss_indir_tbl_max(adapter));

		/*Allow at least 2 queues w/ SR-IOV.*/
		if ((adapter->flags & TXGBE_FLAG_SRIOV_ENABLED) &&
		    (max_queues < 2))
			max_queues = 2;

		/* Verify user input. */
		for (i = 0; i < reta_entries; i++)
			if (indir[i] >= max_queues)
				return -EINVAL;

		if (adapter->flags & TXGBE_FLAG_SRIOV_ENABLED) {
			for (i = 0; i < reta_entries; i++)
				adapter->rss_indir_tbl[i] = indir[i];
			txgbe_store_vfreta(adapter);
		} else {
			for (i = 0; i < reta_entries; i++)
				adapter->rss_indir_tbl[i] = indir[i];
			txgbe_store_reta(adapter);
		}
	}

	if (key) {
		memcpy(adapter->rss_key, key, txgbe_get_rxfh_key_size(netdev));

		if (adapter->flags & TXGBE_FLAG_SRIOV_ENABLED) {
			unsigned int pf_pool = adapter->num_vfs;
			for (i = 0; i < 10; i++)
				wr32(hw, TXGBE_RDB_VMRSSRK(i, pf_pool), adapter->rss_key[i]);
		} else {
			/* Fill out hash function seeds */
			for (i = 0; i < 10; i++)
				wr32(hw, TXGBE_RDB_RSSRK(i), adapter->rss_key[i]);
		}
	}

	return 0;
}
#endif /* ETHTOOL_GRSSH && ETHTOOL_SRSSH */

#ifdef HAVE_ETHTOOL_GET_TS_INFO
#ifdef HAVE_KERNEL_ETHTOOL_TS_INFO
static int txgbe_get_ts_info(struct net_device *dev,
			     struct kernel_ethtool_ts_info *info)
#else
static int txgbe_get_ts_info(struct net_device *dev,
			     struct ethtool_ts_info *info)
#endif
{
	struct txgbe_adapter *adapter = netdev_priv(dev);

	/* we always support timestamping disabled */
	info->rx_filters = 1 << HWTSTAMP_FILTER_NONE;

#ifdef HAVE_PTP_1588_CLOCK

	info->so_timestamping =
		SOF_TIMESTAMPING_TX_SOFTWARE |
		SOF_TIMESTAMPING_RX_SOFTWARE |
		SOF_TIMESTAMPING_SOFTWARE |
		SOF_TIMESTAMPING_TX_HARDWARE |
		SOF_TIMESTAMPING_RX_HARDWARE |
		SOF_TIMESTAMPING_RAW_HARDWARE;

	if (adapter->ptp_clock)
		info->phc_index = ptp_clock_index(adapter->ptp_clock);
	else
		info->phc_index = -1;

	info->tx_types =
		(1 << HWTSTAMP_TX_OFF) |
		(1 << HWTSTAMP_TX_ON);

	info->rx_filters |=
		(1 << HWTSTAMP_FILTER_PTP_V1_L4_SYNC) |
		(1 << HWTSTAMP_FILTER_PTP_V1_L4_DELAY_REQ) |
		(1 << HWTSTAMP_FILTER_PTP_V2_L2_EVENT) |
		(1 << HWTSTAMP_FILTER_PTP_V2_L4_EVENT) |
		(1 << HWTSTAMP_FILTER_PTP_V2_SYNC) |
		(1 << HWTSTAMP_FILTER_PTP_V2_L2_SYNC) |
		(1 << HWTSTAMP_FILTER_PTP_V2_L4_SYNC) |
		(1 << HWTSTAMP_FILTER_PTP_V2_DELAY_REQ) |
		(1 << HWTSTAMP_FILTER_PTP_V2_L2_DELAY_REQ) |
		(1 << HWTSTAMP_FILTER_PTP_V2_L4_DELAY_REQ) |
		(1 << HWTSTAMP_FILTER_PTP_V2_EVENT);

#endif /* HAVE_PTP_1588_CLOCK */
	return 0;
}
#endif /* HAVE_ETHTOOL_GET_TS_INFO */

#endif /* ETHTOOL_GRXRINGS */
#ifdef ETHTOOL_SCHANNELS
static unsigned int txgbe_max_channels(struct txgbe_adapter *adapter)
{
	unsigned int max_combined;
	u8 tcs = netdev_get_num_tc(adapter->netdev);

	if (!(adapter->flags & TXGBE_FLAG_MSIX_ENABLED)) {
		/* We only support one q_vector without MSI-X */
		max_combined = 1;
	} else if (adapter->flags & TXGBE_FLAG_SRIOV_ENABLED) {
		/* SR-IOV currently only allows one queue on the PF */
		max_combined = adapter->ring_feature[RING_F_RSS].mask + 1;
	} else if (tcs > 1) {
		/* For DCB report channels per traffic class */
		if (tcs > 4) {
			/* 8 TC w/ 8 queues per TC */
			max_combined = 8;
		} else {
			/* 4 TC w/ 16 queues per TC */
			max_combined = 16;
		}
	} else if (adapter->atr_sample_rate) {
		/* support up to 64 queues with ATR */
		max_combined = TXGBE_MAX_FDIR_INDICES;
		if (adapter->xdp_prog)
			max_combined = TXGBE_MAX_XDP_RSS_INDICES;
	} else {
		/* support up to max allowed queues with RSS */
		max_combined = txgbe_max_rss_indices(adapter);
	}

	return max_combined;
}

static void txgbe_get_channels(struct net_device *dev,
			       struct ethtool_channels *ch)
{
	struct txgbe_adapter *adapter = netdev_priv(dev);

	/* report maximum channels */
	ch->max_combined = txgbe_max_channels(adapter);

	/* report info for other vector */
	if (adapter->flags & TXGBE_FLAG_MSIX_ENABLED) {
		ch->max_other = NON_Q_VECTORS;
		ch->other_count = NON_Q_VECTORS;
	}

	/* record RSS queues */
	ch->combined_count = adapter->ring_feature[RING_F_RSS].indices;

	/* nothing else to report if RSS is disabled */
	if (ch->combined_count == 1)
		return;

	/* we do not support ATR queueing if SR-IOV is enabled */
	if (adapter->flags & TXGBE_FLAG_SRIOV_ENABLED)
		return;

	/* same thing goes for being DCB enabled */
	if (netdev_get_num_tc(dev) > 1)
		return;

	/* if ATR is disabled we can exit */
	if (!adapter->atr_sample_rate)
		return;

	/* report flow director queues as maximum channels */
	ch->combined_count = adapter->ring_feature[RING_F_FDIR].indices;
}

static int txgbe_set_channels(struct net_device *dev,
			      struct ethtool_channels *ch)
{
	struct txgbe_adapter *adapter = netdev_priv(dev);
	unsigned int count = ch->combined_count;
	u8 max_rss_indices = txgbe_max_rss_indices(adapter);

	/* verify they are not requesting separate vectors */
	if (!count || ch->rx_count || ch->tx_count)
		return -EINVAL;

	/* verify other_count has not changed */
	if (ch->other_count != NON_Q_VECTORS)
		return -EINVAL;

	/* verify the number of channels does not exceed hardware limits */
	if (count > txgbe_max_channels(adapter))
		return -EINVAL;

	if (count < adapter->active_vlan_limited + 1) {
		e_dev_info("vlan rate limit active, can't set less than active "
			   "limited vlan + 1:%d", (adapter->active_vlan_limited + 1));
		return -EINVAL;
	}
	/* update feature limits from largest to smallest supported values */
	adapter->ring_feature[RING_F_FDIR].limit = count;

	/* cap RSS limit */
	if (count > max_rss_indices)
		count = max_rss_indices;
	adapter->ring_feature[RING_F_RSS].limit = count;

#if IS_ENABLED(CONFIG_FCOE)
	/* cap FCoE limit at 8 */
	if (count > TXGBE_RDB_FCRE_TBL_SIZE)
		count = TXGBE_RDB_FCRE_TBL_SIZE;
	adapter->ring_feature[RING_F_FCOE].limit = count;
#endif /* CONFIG_FCOE */

	/* use setup TC to update any traffic class queue mapping */
	return txgbe_setup_tc(dev, netdev_get_num_tc(dev));
}
#endif /* ETHTOOL_SCHANNELS */

#ifdef ETHTOOL_GMODULEINFO
static int txgbe_get_module_info(struct net_device *dev,
				       struct ethtool_modinfo *modinfo)
{
	struct txgbe_adapter *adapter = netdev_priv(dev);
	struct txgbe_hw *hw = &adapter->hw;
	u32 status;
	u8 sff8472_rev, addr_mode;
	u8 identifier = 0;
	u8 sff8636_rev = 0;
	bool page_swap = false;
	u32 swfw_mask = hw->phy.phy_semaphore_mask;
	u32 value;

	if (hw->mac.type == txgbe_mac_aml40) {
		value = rd32(hw, TXGBE_GPIO_EXT);
		if (value & TXGBE_SFP1_MOD_PRST_LS) {
			return -EIO;
		}

		if (!netif_carrier_ok(dev)) {
			e_err(drv, "\"Ethool -m\" is supported only when link is up for 40G.\n");
			return -EIO;
		}
	}

	if (hw->mac.type == txgbe_mac_aml) {
		value = rd32(hw, TXGBE_GPIO_EXT);
		if (value & TXGBE_SFP1_MOD_ABS_LS) {
			return -EIO;
		}
	}

	if (hw->mac.type != txgbe_mac_sp) {
		if (0 != TCALL(hw, mac.ops.acquire_swfw_sync, swfw_mask))
		return -EBUSY;
		
		if (!test_bit(__TXGBE_DOWN, &adapter->state))
			cancel_work_sync(&adapter->sfp_sta_task);

		status = TCALL(hw, phy.ops.read_i2c_eeprom,
							TXGBE_SFF_IDENTIFIER,
							&identifier);
		if (status != 0)
			goto ERROR_IO;

		switch (identifier) {
		case TXGBE_SFF_IDENTIFIER_SFP:
			/* Check whether we support SFF-8472 or not */
			status = TCALL(hw, phy.ops.read_i2c_eeprom,
								TXGBE_SFF_SFF_8472_COMP,
								&sff8472_rev);
			if (status != 0)
				goto ERROR_IO;

			/* addressing mode is not supported */
			status = TCALL(hw, phy.ops.read_i2c_eeprom,
								TXGBE_SFF_SFF_8472_SWAP,
								&addr_mode);
			if (status != 0)
				goto ERROR_IO;

			if (addr_mode & TXGBE_SFF_ADDRESSING_MODE) {
				e_err(drv, "Address change required to access page 0xA2, "
					"but not supported. Please report the module type to the "
					"driver maintainers.\n");
				page_swap = true;
			}

			if (sff8472_rev == TXGBE_SFF_SFF_8472_UNSUP || page_swap ||
				!(addr_mode & TXGBE_SFF_DDM_IMPLEMENTED)) {
				/* We have a SFP, but it does not support SFF-8472 */
				modinfo->type = ETH_MODULE_SFF_8079;
				modinfo->eeprom_len = ETH_MODULE_SFF_8079_LEN;
			} else {
				/* We have a SFP which supports a revision of SFF-8472. */
				modinfo->type = ETH_MODULE_SFF_8472;
				modinfo->eeprom_len = ETH_MODULE_SFF_8472_LEN;
			}
			break;
		case TXGBE_SFF_IDENTIFIER_QSFP:
		case TXGBE_SFF_IDENTIFIER_QSFP_PLUS:
			status = TCALL(hw, phy.ops.read_i2c_eeprom,
								TXGBE_SFF_SFF_REVISION_ADDR,
								&sff8636_rev);
			if (status != 0)
				goto ERROR_IO;

			/* Check revision compliance */
			if (sff8636_rev > 0x02) {
				/* Module is SFF-8636 compliant */
				modinfo->type = ETH_MODULE_SFF_8636;
				modinfo->eeprom_len = TXGBE_MODULE_QSFP_MAX_LEN;
			} else {
				modinfo->type = ETH_MODULE_SFF_8436;
				modinfo->eeprom_len = TXGBE_MODULE_QSFP_MAX_LEN;
			}
			break;
		default:
			e_err(drv, "SFF Module Type not recognized.\n");
			return -EINVAL;
		}

		TCALL(hw, mac.ops.release_swfw_sync, swfw_mask);
	} else {
		modinfo->type = adapter->eeprom_type;
		modinfo->eeprom_len = adapter->eeprom_len;
	}

	return 0;

ERROR_IO:
	TCALL(hw, mac.ops.release_swfw_sync, swfw_mask);
	return -EIO;
}

#define SFF_A2_ALRM_FLG			0x170
#define SFF_A2_WARN_FLG			0x174
#define SFF_A2_TEMP			0x160
#define SFF_A2_RX_PWR			0x169

static int txgbe_get_module_eeprom(struct net_device *dev,
					 struct ethtool_eeprom *ee,
					 u8 *data)
{
	struct txgbe_adapter *adapter = netdev_priv(dev);
	struct txgbe_hw *hw = &adapter->hw;
	int i = 0;
	bool is_sfp = false;
	u32 value;
	u8 identifier = 0;
	u32 swfw_mask = hw->phy.phy_semaphore_mask;
	u8 databyte;
	s32 status = 0;

	if (hw->mac.type == txgbe_mac_aml40) {
		value = rd32(hw, TXGBE_GPIO_EXT);
		if (value & TXGBE_SFP1_MOD_PRST_LS) {
			return -EIO;
		}
	}

	if (hw->mac.type == txgbe_mac_aml) {
		value = rd32(hw, TXGBE_GPIO_EXT);
		if (value & TXGBE_SFP1_MOD_ABS_LS) {
			return -EIO;
		}
	}

	if (hw->mac.type != txgbe_mac_sp) {
		if (0 != TCALL(hw, mac.ops.acquire_swfw_sync, swfw_mask))
			return -EBUSY;

		if (!test_bit(__TXGBE_DOWN, &adapter->state))
			cancel_work_sync(&adapter->sfp_sta_task);

		if (ee->len == 0)
			goto ERROR_INVAL;

		status = TCALL(hw, phy.ops.read_i2c_eeprom,
							TXGBE_SFF_IDENTIFIER,
							&identifier);
		if (status != 0)
			goto ERROR_IO;

		if (identifier == TXGBE_SFF_IDENTIFIER_SFP)
			is_sfp = true;

		memset(data, 0, ee->len);
		for (i = 0; i < ee->len; i++) {
			u32 offset = i + ee->offset;
			u32 page = 0;

			/* I2C reads can take long time */
			if (test_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
				goto ERROR_BUSY;

			if (is_sfp) {
				if (offset < ETH_MODULE_SFF_8079_LEN)
					status = TCALL(hw, phy.ops.read_i2c_eeprom, offset,
								&databyte);
				else
					status = TCALL(hw, phy.ops.read_i2c_sff8472, offset,
								&databyte);

				if (status != 0)
					goto ERROR_IO;
			} else {
				while (offset >= ETH_MODULE_SFF_8436_LEN) {
					offset -= ETH_MODULE_SFF_8436_LEN / 2;
					page++;
				}

				if (page == 0 || !(data[0x2] & 0x4)) {
					status = TCALL(hw, phy.ops.read_i2c_sff8636, page, offset,
								&databyte);

					if (status != 0)
						goto ERROR_IO;
				}
			}
			data[i] = databyte;
		}
	} else {
		if (ee->len == 0)
			goto ERROR_INVAL;

		if (0 != TCALL(hw, mac.ops.acquire_swfw_sync, swfw_mask))
			return -EBUSY;

		/*when down, can't know sfp change, get eeprom from i2c*/
		if (test_bit(__TXGBE_DOWN, &adapter->state)) {
			for (i = ee->offset; i < ee->offset + ee->len; i++) {
				/* I2C reads can take long time */
				if (test_bit(__TXGBE_IN_SFP_INIT, &adapter->state))
					goto ERROR_BUSY;

				if (i < ETH_MODULE_SFF_8079_LEN)
					status = TCALL(hw, phy.ops.read_i2c_eeprom, i,
						&databyte);
				else
					status = TCALL(hw, phy.ops.read_i2c_sff8472, i,
						&databyte);

				if (status != 0)
					goto ERROR_IO;

				data[i - ee->offset] = databyte;
			}
		} else {
			if (adapter->eeprom_type == ETH_MODULE_SFF_8472) {

				cancel_work_sync(&adapter->sfp_sta_task);

				/*alarm flag*/
				for (i = SFF_A2_ALRM_FLG; i <= SFF_A2_ALRM_FLG + 1; i++){
					status = TCALL(hw, phy.ops.read_i2c_sff8472, i,
						&databyte);

					if (status != 0)
						goto ERROR_IO;

					adapter->i2c_eeprom[i] = databyte;
				}
				/*warm flag*/
				for (i = SFF_A2_WARN_FLG; i <= SFF_A2_WARN_FLG + 1; i++){
					status = TCALL(hw, phy.ops.read_i2c_sff8472, i,
						&databyte);

					if (status != 0)
						goto ERROR_IO;

					adapter->i2c_eeprom[i] = databyte;
				}
				/*dom monitor value*/
				for (i = SFF_A2_TEMP; i <= SFF_A2_RX_PWR + 1; i++){
					status = TCALL(hw, phy.ops.read_i2c_sff8472, i,
						&databyte);

					if (status != 0)
						goto ERROR_IO;

					adapter->i2c_eeprom[i] = databyte;
				}
			}
			for (i = ee->offset; i < ee->offset + ee->len; i++)
				data[i - ee->offset] = adapter->i2c_eeprom[i];
		}
	}
	TCALL(hw, mac.ops.release_swfw_sync, swfw_mask);
	return 0;
ERROR_BUSY:
	TCALL(hw, mac.ops.release_swfw_sync, swfw_mask);
	return -EBUSY;
ERROR_IO:
	TCALL(hw, mac.ops.release_swfw_sync, swfw_mask);
	return -EIO;
ERROR_INVAL:
	return -EINVAL;
}
#endif /* ETHTOOL_GMODULEINFO */

#ifdef ETHTOOL_GEEE
#ifdef HAVE_ETHTOOL_KEEE
static int txgbe_get_eee(struct net_device *netdev, struct ethtool_keee *edata)
#else
static int txgbe_get_eee(struct net_device *netdev, struct ethtool_eee *edata)
#endif
{
	return 0;
}
#endif /* ETHTOOL_GEEE */

#ifdef ETHTOOL_SEEE
#ifdef HAVE_ETHTOOL_KEEE
static int txgbe_set_eee(struct net_device *netdev, struct ethtool_keee *edata)
#else
static int txgbe_set_eee(struct net_device *netdev, struct ethtool_eee *edata)
#endif
{
	struct txgbe_adapter *adapter = netdev_priv(netdev);
	struct txgbe_hw *hw = &adapter->hw;
#ifdef HAVE_ETHTOOL_KEEE
	struct ethtool_keee eee_data;
#else
	struct ethtool_eee eee_data;
#endif
	s32 ret_val;

	if (!(hw->mac.ops.setup_eee &&
	    (adapter->flags2 & TXGBE_FLAG2_EEE_CAPABLE)))
		return -EOPNOTSUPP;

#ifdef HAVE_ETHTOOL_KEEE
	memset(&eee_data, 0, sizeof(struct ethtool_keee));
#else
	memset(&eee_data, 0, sizeof(struct ethtool_eee));
#endif

	ret_val = txgbe_get_eee(netdev, &eee_data);
	if (ret_val)
		return ret_val;

	if (eee_data.eee_enabled && !edata->eee_enabled) {
		if (eee_data.tx_lpi_enabled != edata->tx_lpi_enabled) {
			e_dev_err("Setting EEE tx-lpi is not supported\n");
			return -EINVAL;
		}

		if (eee_data.tx_lpi_timer != edata->tx_lpi_timer) {
			e_dev_err("Setting EEE Tx LPI timer is not "
				  "supported\n");
			return -EINVAL;
		}

		if (eee_data.advertised != edata->advertised) {
			e_dev_err("Setting EEE advertised speeds is not "
				  "supported\n");
			return -EINVAL;
		}

	}

	if (eee_data.eee_enabled != edata->eee_enabled) {

		if (edata->eee_enabled)
			adapter->flags2 |= TXGBE_FLAG2_EEE_ENABLED;
		else
			adapter->flags2 &= ~TXGBE_FLAG2_EEE_ENABLED;

		/* reset link */
		if (netif_running(netdev))
			txgbe_reinit_locked(adapter);
		else
			txgbe_reset(adapter);
	}

	return 0;
}
#endif /* ETHTOOL_SEEE */

static int txgbe_set_flash(struct net_device *netdev, struct ethtool_flash *ef)
{
	int ret;
	const struct firmware *fw;
	struct txgbe_adapter *adapter = netdev_priv(netdev);

	ret = request_firmware(&fw, ef->data, &netdev->dev);
	if (ret < 0)
		return ret;

	if (ef->region == 0) {
		ret = txgbe_upgrade_flash(&adapter->hw, ef->region,
							fw->data, fw->size);
	} else {
		if (txgbe_mng_present(&adapter->hw)) {
		ret = txgbe_upgrade_flash_hostif(&adapter->hw, ef->region,
						fw->data, fw->size);
		} else
			ret = -EOPNOTSUPP;
	}

	release_firmware(fw);
	if (!ret)
		dev_info(&netdev->dev,
			 "loaded firmware %s, reboot to make firmware work\n", ef->data);
	return ret;
}


static struct ethtool_ops txgbe_ethtool_ops = {
#ifdef ETHTOOL_GLINKSETTINGS
	.get_link_ksettings = txgbe_get_link_ksettings,
	.set_link_ksettings = txgbe_set_link_ksettings,
#else
	.get_settings		= txgbe_get_settings,
	.set_settings		= txgbe_set_settings,
#endif
#ifdef ETHTOOL_GFECPARAM
	.get_fecparam = txgbe_get_fec_param,
	.set_fecparam = txgbe_set_fec_param,
#endif /* ETHTOOL_GFECPARAM */
	.get_drvinfo            = txgbe_get_drvinfo,
	.get_regs_len           = txgbe_get_regs_len,
	.get_regs               = txgbe_get_regs,
	.get_wol                = txgbe_get_wol,
	.set_wol                = txgbe_set_wol,
	.nway_reset             = txgbe_nway_reset,
	.get_link               = ethtool_op_get_link,
	.get_eeprom_len         = txgbe_get_eeprom_len,
	.get_eeprom             = txgbe_get_eeprom,
	.set_eeprom             = txgbe_set_eeprom,
	.get_ringparam          = txgbe_get_ringparam,
	.set_ringparam          = txgbe_set_ringparam,
	.get_pauseparam         = txgbe_get_pauseparam,
	.set_pauseparam         = txgbe_set_pauseparam,
	.get_msglevel           = txgbe_get_msglevel,
	.set_msglevel           = txgbe_set_msglevel,
#ifndef HAVE_ETHTOOL_GET_SSET_COUNT
	.self_test_count        = txgbe_diag_test_count,
#endif /* HAVE_ETHTOOL_GET_SSET_COUNT */
	.self_test              = txgbe_diag_test,
	.get_strings            = txgbe_get_strings,
#ifndef HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT
#ifdef HAVE_ETHTOOL_SET_PHYS_ID
	.set_phys_id            = txgbe_set_phys_id,
#else
	.phys_id                = txgbe_phys_id,
#endif /* HAVE_ETHTOOL_SET_PHYS_ID */
#endif /* HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT */
#ifndef HAVE_ETHTOOL_GET_SSET_COUNT
	.get_stats_count        = txgbe_get_stats_count,
#else /* HAVE_ETHTOOL_GET_SSET_COUNT */
	.get_sset_count         = txgbe_get_sset_count,
	.get_priv_flags			= txgbe_get_priv_flags,
	.set_priv_flags			= txgbe_set_priv_flags,
#endif /* HAVE_ETHTOOL_GET_SSET_COUNT */
	.get_ethtool_stats      = txgbe_get_ethtool_stats,
#ifdef HAVE_ETHTOOL_GET_PERM_ADDR
	.get_perm_addr          = ethtool_op_get_perm_addr,
#endif

#ifdef HAVE_ETHTOOL_COALESCE_PARAMS_SUPPORT
	.supported_coalesce_params = ETHTOOL_COALESCE_USECS |
								 ETHTOOL_COALESCE_MAX_FRAMES_IRQ |
								 ETHTOOL_COALESCE_USE_ADAPTIVE,
#endif
	.get_coalesce           = txgbe_get_coalesce,
	.set_coalesce           = txgbe_set_coalesce,
#ifndef HAVE_NDO_SET_FEATURES
	.get_rx_csum            = txgbe_get_rx_csum,
	.set_rx_csum            = txgbe_set_rx_csum,
	.get_tx_csum            = ethtool_op_get_tx_csum,
	.set_tx_csum            = txgbe_set_tx_csum,
	.get_sg                 = ethtool_op_get_sg,
	.set_sg                 = ethtool_op_set_sg,
#ifdef NETIF_F_TSO
	.get_tso                = ethtool_op_get_tso,
	.set_tso                = txgbe_set_tso,
#endif
#ifdef ETHTOOL_GFLAGS
	.get_flags              = ethtool_op_get_flags,
	.set_flags              = txgbe_set_flags,
#endif
#endif /* HAVE_NDO_SET_FEATURES */
#ifdef ETHTOOL_GRXRINGS
	.get_rxnfc              = txgbe_get_rxnfc,
	.set_rxnfc              = txgbe_set_rxnfc,
#ifdef ETHTOOL_SRXNTUPLE
	.set_rx_ntuple          = txgbe_set_rx_ntuple,
#endif
#endif /* ETHTOOL_GRXRINGS */
#ifndef HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT

#ifdef ETHTOOL_GEEE
	.get_eee                = txgbe_get_eee,
#endif /* ETHTOOL_GEEE */
#ifdef ETHTOOL_SEEE
	.set_eee                = txgbe_set_eee,
#endif /* ETHTOOL_SEEE */
#ifdef ETHTOOL_SCHANNELS
	.get_channels           = txgbe_get_channels,
	.set_channels           = txgbe_set_channels,
#endif
#ifdef ETHTOOL_GMODULEINFO
	.get_module_info        = txgbe_get_module_info,
	.get_module_eeprom      = txgbe_get_module_eeprom,
#endif
#ifdef HAVE_ETHTOOL_GET_TS_INFO
	.get_ts_info            = txgbe_get_ts_info,
#endif
#if defined(ETHTOOL_GRSSH) && defined(ETHTOOL_SRSSH)
	.get_rxfh_indir_size	= txgbe_rss_indir_size,
	.get_rxfh_key_size		= txgbe_get_rxfh_key_size,
	.get_rxfh		= txgbe_get_rxfh,
	.set_rxfh		= txgbe_set_rxfh,
#endif /* ETHTOOL_GRSSH && ETHTOOL_SRSSH */
#endif /* HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT */
	.flash_device      	= txgbe_set_flash,
};

#ifdef HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT
static const struct ethtool_ops_ext txgbe_ethtool_ops_ext = {
	.size                   = sizeof(struct ethtool_ops_ext),
	.get_ts_info            = txgbe_get_ts_info,
	.set_phys_id            = txgbe_set_phys_id,
	.get_channels           = txgbe_get_channels,
	.set_channels           = txgbe_set_channels,
#ifdef ETHTOOL_GMODULEINFO
	.get_module_info        = txgbe_get_module_info,
	.get_module_eeprom      = txgbe_get_module_eeprom,
#endif
#if defined(ETHTOOL_GRSSH) && defined(ETHTOOL_SRSSH)
	.get_rxfh_indir_size	= txgbe_rss_indir_size,
	.get_rxfh_key_size	= txgbe_get_rxfh_key_size,
	.get_rxfh		= txgbe_get_rxfh,
	.set_rxfh		= txgbe_set_rxfh,
#endif /* ETHTOOL_GRSSH && ETHTOOL_SRSSH */
#ifdef ETHTOOL_GEEE
	.get_eee                = txgbe_get_eee,
#endif /* ETHTOOL_GEEE */
#ifdef ETHTOOL_SEEE
	.set_eee                = txgbe_set_eee,
#endif /* ETHTOOL_SEEE */
};

#endif /* HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT */
void txgbe_set_ethtool_ops(struct net_device *netdev)
{
#ifndef ETHTOOL_OPS_COMPAT
	netdev->ethtool_ops = &txgbe_ethtool_ops;
#else
	SET_ETHTOOL_OPS(netdev, &txgbe_ethtool_ops);
#endif

#ifdef HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT
	set_ethtool_ops_ext(netdev, &txgbe_ethtool_ops_ext);
#endif /* HAVE_RHEL6_ETHTOOL_OPS_EXT_STRUCT */
}
#endif /* SIOCETHTOOL */

